<?php
/**
 * @package     Joomla.Administrator
 * @subpackage  com_taxibooking
 *
 * @copyright   Copyright (C) 2005 - 2014 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die('Restricted access');

jimport('joomla.application.component.controller');

// new pdf 
require(dirname(dirname(__FILE__))."/dompdf/autoload.inc.php");

use Dompdf\Dompdf;
/**
 * Component Item Controller
 */
class TaxiBookingControllerOrders extends TaxiBookingController
{
	/**
	 * @var		string	The prefix to use with controller messages.
	 * @since	1.6
	 */
	protected $text_prefix = 'COM_TAXIBOOKING';
	/**
	 * The context for storing internal data, e.g. record.
	 *
	 * @var    string
	 * @since  12.2
	 */
	protected $context;
	/**
	 * The URL option for the component.
	 *
	 * @var    string
	 * @since  12.2
	 */
	protected $option;

	/**
	 * The URL view item variable.
	 *
	 * @var    string
	 * @since  12.2
	 */
	protected $view_item;

	/**
	 * The URL view list variable.
	 *
	 * @var    string
	 * @since  12.2
	 */
	protected $view_list;
	/**
	 * Constructor
	 *
	 * @since 1.0
	 */
	function __construct()
	{
		parent::__construct();

		$this->context = 'order';
		// Guess the option as com_NameOfController
		if (empty($this->option))
		{
			$this->option = 'com_taxibooking';
		}
		// Guess the item view as the context.
		if (empty($this->view_item))
		{
			$this->view_item = 'order';
		}
		// Guess the list view as the plural of the item view.
		if (empty($this->view_list))
		{
			$this->view_list = 'orders';
		}
	}

	/**
	 * Method to add a new record.
	 *
	 * @return  mixed  True if the record can be added, a error object if not.
	 *
	 * @since   12.2
	 */
	public function add()
	{
		$app = JFactory::getApplication();
		$context = "$this->option.edit.$this->context";

		// Access check.
		if (!$this->allowAdd())
		{
			// Set the internal error and also the redirect error.
			$this->setError(JText::_('JLIB_APPLICATION_ERROR_CREATE_RECORD_NOT_PERMITTED'));
			$this->setMessage($this->getError(), 'error');

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_list
					. $this->getRedirectToListAppend(), false
				)
			);

			return false;
		}

		// Clear the record edit information from the session.
		$app->setUserState($context . '.data', null);

		// Redirect to the edit screen.
		$this->setRedirect(
			JRoute::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_item
				. $this->getRedirectToItemAppend(), false
			)
		);

		return true;
	}

	/**
	 * Method to check if you can add a new record.
	 *
	 * Extended classes can override this if necessary.
	 *
	 * @param   array  $data  An array of input data.
	 *
	 * @return  boolean
	 *
	 * @since   12.2
	 */
	protected function allowAdd($data = array())
	{
		$user = JFactory::getUser();
		return ($user->authorise('core.create', $this->option));
	}

	/**
	 * Method to cancel an edit.
	 *
	 * @param   string  $key  The name of the primary key of the URL variable.
	 *
	 * @return  boolean  True if access level checks pass, false otherwise.
	 *
	 * @since   12.2
	 */
	public function cancel()
	{
		JSession::checkToken() or jexit(JText::_('JINVALID_TOKEN'));

		$app = JFactory::getApplication();
		$model = $this->getModel('order');
		$table = $model->getTable();
		$checkin = property_exists($table, 'checked_out');
		$context = "$this->option.edit.$this->context";

		if (empty($key))
		{
			$key = $table->getKeyName();
		}

		$recordId = $app->input->getInt($key);

		// Attempt to check-in the current record.
		if ($recordId)
		{
			if ($checkin)
			{
				if ($model->checkin($recordId) === false)
				{
					// Check-in failed, go back to the record and display a notice.
					$this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_CHECKIN_FAILED', $model->getError()));
					$this->setMessage($this->getError(), 'error');

					$this->setRedirect(
						JRoute::_(
							'index.php?option=' . $this->option . '&view=' . $this->view_item
							. $this->getRedirectToItemAppend($recordId, $key), false
						)
					);

					return false;
				}
			}
		}

		// Clean the session data and redirect.
		$this->releaseEditId($context, $recordId);
		$app->setUserState($context . '.data', null);

		$this->setRedirect(
			JRoute::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list
				. $this->getRedirectToListAppend(), false
			)
		);

		return true;
	}

	/**
	 * Logic to create the view for order details
	 *
	 * @access public
	 * @return void
	 * @since 1.0
	 */
	public function show()
	{
		$app   = JFactory::getApplication();
		$model = $this->getModel('order');
		$table = $model->getTable();
		$cid	= JRequest::getVar( 'cid', array(0), 'get', 'array' );
		$context = "$this->option.show.$this->context";

		JRequest::setVar( 'layout', 'default' );

		// Determine the name of the primary key for the data.
		if (empty($key))
		{
			$key = $table->getKeyName();
		}

		// To avoid data collisions the urlVar may be different from the primary key.
		if (empty($urlVar))
		{
			$urlVar = $key;
		}

		// Get the previous record id (if any) and the current record id.
		$recordId = (int) (count($cid) ? $cid[0] : $app->input->getInt($urlVar));

		$this->setRedirect(
			JRoute::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_item
				. $this->getRedirectToItemAppend($recordId, $urlVar), false
			)
		);

		return true;
	}

	/**
	 * Method to check if you can add a new record.
	 *
	 * Extended classes can override this if necessary.
	 *
	 * @param   array   $data  An array of input data.
	 * @param   string  $key   The name of the key for the primary key; default is id.
	 *
	 * @return  boolean
	 *
	 * @since   12.2
	 */
	protected function allowEdit($data = array(), $key = 'id')
	{
		return JFactory::getUser()->authorise('core.edit', 'com_taxibooking');
	}

	/**
	 * Logic to create the view for edit order
	 *
	 * @access public
	 * @return void
	 * @since 1.0
	 */
	public function edit($key = null, $urlVar = null)
	{
		$app   = JFactory::getApplication();
		$model = $this->getModel('order');
		$table = $model->getTable();
		$cid	= JRequest::getVar( 'cid', array(0), 'post', 'array' );
		$context = "$this->option.edit.$this->context";

		// Determine the name of the primary key for the data.
		if (empty($key))
		{
			$key = $table->getKeyName();
		}

		// To avoid data collisions the urlVar may be different from the primary key.
		if (empty($urlVar))
		{
			$urlVar = $key;
		}

		// Get the previous record id (if any) and the current record id.
		$recordId = (int) (count($cid) ? $cid[0] : $app->input->getInt($urlVar));
		$checkin = property_exists($table, 'checked_out');

		// Access check.
		if (!$this->allowEdit(array($key => $recordId), $key))
		{
			$this->setError(JText::_('JLIB_APPLICATION_ERROR_EDIT_NOT_PERMITTED'));
			$this->setMessage($this->getError(), 'error');

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_list
					. $this->getRedirectToListAppend(), false
				)
			);

			return false;
		}

		// Attempt to check-out the new record for editing and redirect.
		if ($checkin && !$model->checkout($recordId))
		{
			// Check-out failed, display a notice but allow the user to see the record.
			$this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_CHECKOUT_FAILED', $model->getError()));
			$this->setMessage($this->getError(), 'error');

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_item
					. $this->getRedirectToItemAppend($recordId, $urlVar), false
				)
			);

			return false;
		}
		else
		{
			// Check-out succeeded, push the new record id into the session.
			$this->holdEditId($context, $recordId);
			$app->setUserState($context . '.data', null);

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_item
					. $this->getRedirectToItemAppend($recordId, $urlVar), false
				)
			);

			return true;
		}
	}

        /**
	 * Logic to save an order
	 *
	 * @access public
	 * @return void
	 * @since 1.0
	 */
	public function save($key = null, $urlVar = null)
	{
		// Check for request forgeries.
		JSession::checkToken() or jexit(JText::_('JINVALID_TOKEN'));

		$app   = JFactory::getApplication();
		$db =  JFactory::getDBO();
		$date = JFactory::getDate();
		$model = $this->getModel('order');
		$table = $model->getTable();
		$post = JRequest::get( 'post' );
		// echo "<pre>";
		// print_r($post);
		// die;
		$task = $this->getTask();
		$elsettings 	= booking_helper::config();
		$lang =  JFactory::getLanguage();
		$lang_tag = $lang->getTag();

		// Determine the name of the primary key for the data.
		if (empty($key))
		{
			$key = $table->getKeyName();
		}

		// To avoid data collisions the urlVar may be different from the primary key.
		if (empty($urlVar))
		{
			$urlVar = $key;
		}

		$recordId = JRequest::getInt($urlVar);

		if ( $model->save($post) ) {

			JRequest::setVar( 'layout', 'default' );

			$recordId = $model->getState($this->context . '.id');
			$this->holdEditId($this->context, $recordId);
			$app->setUserState($this->context . '.data', null);
			$model->checkout($recordId);

			$order_id = $recordId;
			$row_queue = booking_helper::get_order_by_id($order_id);

			// language on which order was submitted, will be used at backend also
			$lang_tag = ($row_queue->language=='*') ? 'en-GB' : $row_queue->language;
			$lang->load('com_taxibooking', JPATH_ADMINISTRATOR, 'en-GB', true);
			$lang->load('com_taxibooking', JPATH_ADMINISTRATOR, $lang_tag, true);

			// send email + sms, generate invoice
			$states = array(1 => JText::_('ACCEPTED'),
				0 => JText::_('REJECTED'),
				-1 => JText::_('ARCHIVED'),
				-2 => JText::_('WAITING'));

			// send email to owner if anything is changed in order
			if ($post['changed']==1)
			{
				$SiteName = $app->getCfg('sitename');
				$MailFrom = $app->getCfg('mailfrom');
				$FromName = $app->getCfg('fromname');
				$FromName = JText::sprintf("NEW_ORDER_EMAIL_YOUR_WEBSITE_NAME", $FromName);

				if($post['state'] == -1) // Archived email
				{
					booking_helper::sendOrderArchiveEmail($row_queue, $elsettings);
				}
				else {
					if ($post['state'] == 1) {
						$row_queue->order_status .= '<span style="color:green;">'.JText::_('NEW_ORDER_STATUS_ACCEPTED').'</span>';
					} elseif ($post['state'] == 0) {
						$row_queue->order_status .= '<span style="color:red;">'.JText::_('NEW_ORDER_STATUS_REJECTED').'</span>';
					} elseif ($post['state'] == -1) {
						$row_queue->order_status .= '<span style="color:yellow;">'.JText::_('NEW_ORDER_STATUS_ARCHIVED').'</span>';
					} elseif ($post['state'] == -2) {
						$row_queue->order_status .= '<span style="color:green;">'.JText::_('NEW_ORDER_STATUS_WAITING').'</span>';
					}

					$payment_data = '';
					JPluginHelper::importPlugin('tbpayment');
					$_dispatcher = JDispatcher::getInstance();
					$_returnValues = $_dispatcher->trigger('plgTbOnShowOrderEmailsInvoice',array( $row_queue->id,$row_queue->payment));
					foreach ($_returnValues as $_returnValue) {
						if ($_returnValue !== null) {
							$payment_data .= $_returnValue;
						}
					}

					$converted_currency_label = '';
					$returnValues = $_dispatcher->trigger('plgTbonShowConvertedPriceLabel', array(&$row_queue->payment, $row_queue->cprice, &$converted_currency_label));

					// glean language specific header and footer info
					$header_info = $elsettings->header_info;
					$header_info = isset($header_info[$lang_tag]) ? changeEditorImageUrl($header_info[$lang_tag]) : '';

					$contact_info = $elsettings->contact_info;
					$contact_info = isset($contact_info[$lang_tag]) ? changeEditorImageUrl($contact_info[$lang_tag]) : '';

					if($row_queue->driver_id > 0){
						$driverObj = booking_helper::getDriverInfo($row_queue->driver_id);
					}

					// now get the email template
					ob_start();
					include( JPATH_COMPONENT_SITE.DS .'templates'.DS.'order_emails'.DS.'confirmation_email.tpl.php' );
					$mailbody = ob_get_contents();
					ob_end_clean();

					$mail =  JFactory::getMailer();

					// Creating a pdf invoice if status is changed to Accepted
					$pdf_file = '';
					if ($post['state'] == 1)
					{
						$header_info = $elsettings->header_info;
						$header_info_pdf = isset($header_info[$lang_tag]) ? changeEditorImagePath($header_info[$lang_tag]) : '';

						$contact_info = $elsettings->contact_info;
						$contact_info_pdf = isset($contact_info[$lang_tag]) ? changeEditorImagePath($contact_info[$lang_tag]) : '';

						ob_start();
						include( JPATH_COMPONENT_SITE.DS .'templates'.DS.'order_invoice.tpl.php' );
						$invoice_template = ob_get_contents();
						ob_end_clean();

						// require_once(dirname(dirname(__FILE__))."/classes/dompdf_lib.php");
						// $dompdf_lib = new Dompdf_lib;
						// $dompdf_lib->set_custom_paper('A4');
						// $pdfoutput = $dompdf_lib->convert_html_to_pdf($invoice_template, '', false);

						$invoice_title = ($elsettings->invoice_title!="") ? $elsettings->invoice_title."_".$order_id.".pdf" : "taxi_reservation_$order_id.pdf";

						$pdf_file = JPATH_COMPONENT . DS . "documents".DS.$invoice_title;
						// file_put_contents($pdf_file, $pdfoutput);

						// fix pdf -> switch to upgraded version of dompdf
						$dompdf = new Dompdf();

						$optionsPdf = $dompdf->getOptions();
						$optionsPdf->setIsHtml5ParserEnabled('true');
						$optionsPdf->isRemoteEnabled('true');
						$dompdf->setOptions($optionsPdf);
						$dompdf->loadHtml($invoice_template);

						// (Optional) Setup the paper size and orientation
						$dompdf->setPaper('A4', 'portrait');

						// Render the HTML as PDF
						$dompdf->render();

						file_put_contents($pdf_file, $dompdf->output());

						// update order invoice title
						$query = 'UPDATE #__taxibooking_orders SET'
							.' invoice_title =' . $db->Quote($invoice_title)
							.' WHERE id = ' . (int)$order_id;
						$db->setQuery($query);
						$db->query();

						// confirm booked seats if this is shuttle booking
						if($row_queue->booking_type=='shuttle'){
							booking_helper::confirm_shuttle_booking($row_queue);
						}

						// send sms
						if((int)$elsettings->send_user_sms==1 && $row_queue->phone!="")
						{
							// get the order info
							$query = 'SELECT o.*, pm.title AS payment_method
								FROM #__taxibooking_orders AS o
								LEFT JOIN #__taxibooking_paymentmethods AS pm ON pm.id = o.payment
								WHERE o.id = ' . (int)$order_id;
							$db->setQuery($query);
							$db->query();
							$order_info = $db->loadObject();

							$response = booking_helper::send_order_sms($order_info, $elsettings);
							$app->enqueueMessage($response, 'notice');
						}

						// recreate google event if status is changed from Rejected -> Accepted
						//$order_car = booking_helper::get_car_details($row_queue->vehicletype);
						//if($order_car->google_calendar_enabled==1 && $order_car->use_as_shuttle!=2)
						//{
						    //$event_id = booking_helper::createCalendarEvent($row_queue,$order_car,$order_car_rel_id,$booking_time_start,$booking_time_end);
						//}
					}

					$mail->setSender(array($MailFrom, $FromName));
					$mail->setSubject(JText::sprintf('BOOKING_STATUS_CHANGED_ORDER', $row_queue->order_number));
					$mail->setBody($mailbody);
					if($pdf_file!=""&&$elsettings->show_price==1&&strpos(strtolower($row_queue->custom_payment), 'jayride')===false)
					{
					    $mail->addAttachment($pdf_file);
					}
					$mail->IsHTML(true);
					$mail->addRecipient($row_queue->email);
					$sent = $mail->Send();

					// Sending mail to driver if driver is selected
					if ($post['driver_id'] > 0 && ($post['driver_id'] != $post['old_driver']))
					{
						$query = 'SELECT name, email FROM #__users WHERE id = ' . $db->Quote($post['driver_id']);
						$db->setQuery($query);
						$db->query();
						$driver_obj = $db->loadObject();
						if($driver_obj)
						{
						    $driver_name = $driver_obj->name;
						    $driver_email = $driver_obj->email;

						    $receiver='driver';
						// now get the email template
						ob_start();
						include( JPATH_COMPONENT_SITE.DS .'templates'.DS.'order_emails'.DS.'admin_email.tpl.php' );
						$mailbody = ob_get_contents();
						ob_end_clean();

						    $mail->ClearAllRecipients();
						    $mail->ClearAttachments();
						    $mail->setSender(array($MailFrom, $FromName));
						    $mail->setSubject(JText::sprintf('DRIVER_BOOKING_NOTIFICATION', $driver_email) .' '.JText::sprintf('BOOKING_STATUS_CHANGED_ORDER', $row_queue->order_number));
						    $mail->setBody($mailbody);
						    if($pdf_file!=""&&$elsettings->show_price==1)
						    {
							$mail->addAttachment($pdf_file);
						    }
						    $mail->IsHTML(true);
						    $mail->addRecipient($driver_email);
						    $sent = $mail->Send();
						}
					}

					// Remove car booking data if order status is changed to rejected
					if ($post['state'] == 0 && $row_queue->state != $post['old_state']){
						booking_helper::cancel_car_bookings($order_id);
					}

					// changed order status is not Accpeted so, remove shuttle booking
					if($row_queue->booking_type=='shuttle' && $row_queue->state!=1){
						booking_helper::remove_shuttle_booking($order_id);
					}

					// if this is user is not registered yet, create an account silently
					$this->_createUserAccount($row_queue);
				}
			}

			// Redirect back to the details screen.
			$this->setMessage(JText::_('COM_TAXIBOOKING_ORDER_SAVE_SUCCESS'));
			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_item
					. $this->getRedirectToItemAppend($recordId, $urlVar), false
				)
			);

		} else {
			JRequest::setVar( 'layout', 'edit' );

			// Redirect back to the edit screen.
			$this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_SAVE_FAILED', $model->getError()));
			$this->setMessage($this->getError(), 'error');

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_item
					. $this->getRedirectToItemAppend($recordId, $urlVar), false
				)
			);

			return false;
		}
	}

	/**
	* Create user account silently
	*
	* @access	public
	* @since	1.0
	*/
	private function _createUserAccount($order)
	{
		//initialise variables
		$app = JFactory::getApplication();
		$db =  JFactory::getDBO();
		$elsettings =  booking_helper::config();
		$config = JFactory::getConfig();
		$user_params	= JComponentHelper::getParams('com_users');

		// Initialise the table with JUser.
		$user = new JUser;

		// Prepare the data for the user object.
		$data = array();
		$data['name']	        = $order->names;
		$data['username']	= $order->email;
		$data['email']		= $order->email;
		$data['password']	= JUserHelper::genRandomPassword();
		$data['activation'] = '';
		$data['block'] = 0;

		// Get the groups the user should be added to after registration.
		$data['groups'] = array();

		// Get the default new user group, Registered if not specified.
		$data['groups'][] = $user_params->get('new_usertype', 2);

		// Bind the data.
		if (!$user->bind($data)) {
		    return false;
		}

		// Store the data.
		if (!$user->save()) {
		    return false;
		}

		// Compile the notification mail values.
		$data['fromname']	= $config->get('fromname');
		$data['mailfrom']	= $config->get('mailfrom');
		$data['sitename']	= $config->get('sitename');
		$data['siteurl']	= JUri::root();
		$data['password_clear']	= $user->password_clear;

		$emailSubject	= JText::sprintf(
			'COM_TAXIBOOKING_EMAIL_ACCOUNT_DETAILS',
			$data['name'],
			$data['sitename']
		);

		$emailBody = JText::sprintf(
			'COM_TAXIBOOKING_EMAIL_REGISTERED_BODY',
			$data['name'],
			$data['sitename'],
			$data['siteurl'],
			$data['username'],
			$data['password_clear']
		);

		// Send the registration email.
		$return = JFactory::getMailer()->sendMail($data['mailfrom'], $data['fromname'], $data['email'], $emailSubject, $emailBody);

		//Send Notification mail to administrators
		$emailSubject = JText::sprintf(
			'COM_TAXIBOOKING_EMAIL_ACCOUNT_DETAILS',
			$data['name'],
			$data['sitename']
		);

		$emailBodyAdmin = JText::sprintf(
			'COM_TAXIBOOKING_EMAIL_REGISTERED_NOTIFICATION_TO_ADMIN_BODY',
			$data['name'],
			$data['username'],
			$data['siteurl']
		);

		// get all admin users
		$query = 'SELECT name, email, sendEmail' .
			' FROM #__users' .
			' WHERE sendEmail=1';
		$db->setQuery( $query );
		$rows = $db->loadObjectList();

		// Send mail to all superadministrators id
		foreach( $rows as $row )
		{
		    $return = JFactory::getMailer()->sendMail($data['mailfrom'], $data['fromname'], $row->email, $emailSubject, $emailBodyAdmin);
		    // Check for an error.
		    if ($return !== true) {
			return false;
		    }
		}

		return true;
	}

	/**
	 * Exports selected items in CSV.
	 */
	public function export()
	{
		// Check for request forgeries
		JSession::checkToken() or die(JText::_('JINVALID_TOKEN'));

		// Get items to remove from the request.
		$cid = JFactory::getApplication()->input->get('cid', array(), 'array');

		if (!is_array($cid) || count($cid) < 1)
		{
			JLog::add(JText::_($this->text_prefix . '_NO_ITEM_SELECTED'), JLog::WARNING, 'jerror');
			$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false));
		}
		else
		{
			$filter_order = JRequest::setVar( 'filter_order', 'filter_order' );
			$filter_order_dir = JRequest::setVar( 'filter_order_Dir', 'filter_order_Dir' );

			// Get the model.
			$model = $this->getModel('orders');

			// Make sure the item ids are integers
			jimport('joomla.utilities.arrayhelper');
			JArrayHelper::toInteger($cid);

			$db 	= JFactory::getDBO();
			$query = $db->getQuery(true);

			// Select the required fields from the table.
			$query->select(
				$model->getState( 'list.select', 'a.*' )
			);
			$query->from($db->quoteName('#__taxibooking_orders') . ' AS a');

			 // Join over the driver
			$query->select('u.name AS drivername')
				->join('LEFT', $db->quoteName('#__users') . ' AS u ON u.id = a.driver_id');

			// Join over the payment methods
			$query->select('pm.title AS payment_method')
				->join('LEFT', $db->quoteName('#__taxibooking_paymentmethods') . ' AS pm ON pm.id = a.payment');

			$query->where('a.id IN ( ' . implode(',', $cid) . ')');

			// Add the list ordering clause.
			$orderCol = $model->getState('list.ordering', 'a.datetime1');
			$orderDirn = $model->getState('list.direction', 'DESC');
			$query->order($db->escape($orderCol . ' ' . $orderDirn));

			$db->setQuery($query);
			$db->query();
			$rows = $db->loadObjectList();

			ob_end_clean();

			$file_name = 'taxibooking_order_export.csv';

			header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
			header('Accept-Ranges: bytes');
			header('Content-Disposition: attachment; filename='.basename($file_name).';');
			header('Content-Type: text/plain; '.'_ISO');
			header('Expires: ' . gmdate('D, d M Y H:i:s') . ' GMT');
			header('Pragma: no-cache');

			// create a file pointer connected to the output stream
			$output = fopen('php://output', 'w');

			// output the column headings
			$csv_header = array('CUSTOMER NAME','EMAIL','PHONE','FROM','TO','PICKUP DATE','DURATION','RETURN DATE','CAR','DRIVER','PASSENGERS','SUITCASES','INFANT SEATS','CHILD SEATS','BOOSTER SEATS','PRICE','PAYMENT','ID','STATUS');

			fputcsv($output, $csv_header);

			for ($i = 0, $n = count($rows); $i < $n; $i++) {
				$row = $rows[$i];
				if ($row->state == 1) {
				    $state = JText::_('ACCEPTED');
				} else if ($row->state == 0) {
				    $state = JText::_('REJECTED');
				} else if ($row->state == -1) {
				    $state = JText::_('ARCHIVED');
				} else if ($row->state == -2) {
				    $state = JText::_('WAITING');
				}

				$csv_array = array($row->names,
						   $row->email,
						   $row->phone,
						    $row->begin,
						    $row->end,
						   booking_helper::get_order_pickup_date($row),
						   $row->duration_text,
						   booking_helper::get_order_return_date($row),
						   booking_helper::get_order_car($row),
						   $row->drivername,
						   $row->selpassengers,
						   $row->selluggage,
						   $row->selinfantseats,
						   $row->selchildseats,
						   $row->selboosterseats,
						    number_format($row->cprice, 2, '.', ','),
						    booking_helper::get_order_payment($row),
						    $row->id,
						    $state,
						    );
				fputcsv($output, $csv_array);
			}
			exit();
		}
	}

	/**
	 * Removes an item.
	 *
	 * @return  void
	 *
	 * @since   12.2
	 */
	public function remove()
	{
		// Check for request forgeries
		JSession::checkToken() or die(JText::_('JINVALID_TOKEN'));

		// Get items to remove from the request.
		$cid = JFactory::getApplication()->input->get('cid', array(), 'array');

		if (!is_array($cid) || count($cid) < 1)
		{
			JLog::add(JText::_($this->text_prefix . '_NO_ITEM_SELECTED'), JLog::WARNING, 'jerror');
		}
		else
		{
			// Get the model.
			$model = $this->getModel('orders');

			// Make sure the item ids are integers
			jimport('joomla.utilities.arrayhelper');
			JArrayHelper::toInteger($cid);

			// Remove the items.
			if ($model->delete($cid))
			{
				$this->setMessage(JText::plural($this->text_prefix . '_N_ITEMS_DELETED', count($cid)));
			}
			else
			{
				$this->setMessage($model->getError());
			}
		}

		$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false));
	}
	/**
	 * Logic to download invoice for an order
	 *
	 * @access public
	 * @return void
	 * @since 1.0
	 */
	function pdf_invoice()
	{
		$db 	= JFactory::getDBO();
		$cid	= JRequest::getInt( 'cid', 0 );

		if($cid>0)
		{
			// get the order info
			$query = 'SELECT o.*
				FROM #__taxibooking_orders AS o
				WHERE o.id = ' . $cid;
			$db->setQuery($query);
			$db->query();
			$order = $db->loadObject();

			$invoice_file = JPATH_BASE.DS.'components'.DS.'com_taxibooking'.DS.'documents'.DS.$order->invoice_title;
			booking_helper::outputStoredPDF($invoice_file);
		}
		exit();
	}

	function order_invoice()
	{
		$order_id		= JRequest::getInt( 'oid', 0);
		$app = JFactory::getApplication();
		$db = JFactory::getDBO();
		$site_name = $app->getCfg('sitename');

		$mainframe = JFactory::getApplication();
		// Get the page/component configuration
		$params =  JComponentHelper::getParams('com_taxibooking');
		$elsettings =  booking_helper::config();
		$lang =  JFactory::getLanguage();
		$lang_tag = $lang->getTag();

		//Get mailinformation
		$SiteName = $mainframe->getCfg('sitename');
		$MailFrom = $mainframe->getCfg('mailfrom');
		$FromName = $mainframe->getCfg('fromname');
		$tzoffset = $mainframe->getCfg('offset');

		// get the order info
		$query = 'SELECT o.*, pm.title AS payment_method
			FROM #__taxibooking_orders AS o
			LEFT JOIN #__taxibooking_paymentmethods AS pm ON pm.id = o.payment
			WHERE o.id = ' . $order_id;
		$db->setQuery($query);
		$db->query();
		$row_queue = $db->loadObject();

		$row_queue->order_status .= '<span style="color:green;">'.JText::_('NEW_ORDER_STATUS_ACCEPTED').'</span>';

		// glean language specific header and footer info
		$header_info = $elsettings->header_info;
		$temp = $header_info[$lang_tag];
		$header_info = changeEditorImageUrl($temp);
		$header_info_pdf = changeEditorImagePath($temp);

		$contact_info = $elsettings->contact_info;
		$temp = $contact_info[$lang_tag];
		$contact_info = changeEditorImageUrl($temp);
		$contact_info_pdf = changeEditorImagePath($temp);

		$payment_data = '';
		JPluginHelper::importPlugin('tbpayment');
		$dispatcher = JDispatcher::getInstance();
		$_returnValues = $dispatcher->trigger('plgTbOnShowOrderEmailsInvoice',array( $row_queue->id,$row_queue->payment));
		foreach ($_returnValues as $_returnValue) {
			if ($_returnValue !== null) {
				$payment_data .= $_returnValue;
			}
		}

		$payment_labels = '';
		$returnValues = $dispatcher->trigger('plgTbonSelectedShowPriceLabel', array(&$row_queue->payment, $row_queue->cprice, &$payment_labels));

		$converted_currency_label = '';
		$returnValues = $dispatcher->trigger('plgTbonShowConvertedPriceLabel', array(&$row_queue->payment, $row_queue->cprice, &$converted_currency_label));

		ob_start();
		include( JPATH_COMPONENT_SITE.DS .'templates'.DS.'order_invoice.tpl.php' );
		$invoice_template = ob_get_contents();
		ob_end_clean();
		die(print $invoice_template);
	}

	public function send_notes()
	{
		// Check for request forgeries
		JSession::checkToken() or die(JText::_('JINVALID_TOKEN'));

		$app = JFactory::getApplication();
		$config = JFactory::getConfig();
		$date = JFactory::getDate();
		$user = JFactory::getUser();
		$db = JFactory::getDBO();
		$lang =  JFactory::getLanguage();
		$lang_tag = $lang->getTag();
		$elsettings 	= booking_helper::config();

		$order_id	= JRequest::getInt( 'id', 0 );
		$admin_notes = JRequest::getVar( 'admin_notes', '', 'post', 'string', JREQUEST_ALLOWRAW );

		// glean language specific header and footer info
		$header_info = $elsettings->header_info;
		$header_info = isset($header_info[$lang_tag]) ? $header_info[$lang_tag] : '';

		$contact_info = $elsettings->contact_info;
		$contact_info = isset($contact_info[$lang_tag]) ? $contact_info[$lang_tag] : '';

		if($order_id>0 && $admin_notes!="")
		{
			$order_notes = new stdClass();
			$order_notes->order_id = $order_id;
			$order_notes->text = $admin_notes;
			$order_notes->created_by = $user->get('id');
			$order_notes->created_date = $date->toSql();

			$db->insertObject('#__taxibooking_order_notes', $order_notes);

			// send notification to customer
			$order = booking_helper::get_order_by_id($order_id);
			$mail_body = JText::sprintf('ORDER_EDIT_ADMIN_NOTES_INTRO', $order->names);
			$mail_body .= "<br/><br/>";
			$mail_body.= changeEditorImageUrl($admin_notes);
			$mail_body .= "<br/><br/>";
			$mail_body .= $user->get('name')."<br/>";
			$mail_body .= $contact_info;

			$mail =  JFactory::getMailer();
			$SiteName = $app->getCfg('sitename');
			$MailFrom = $app->getCfg('mailfrom');
			$FromName = $app->getCfg('fromname');
			$FromName = JText::sprintf("NEW_ORDER_EMAIL_YOUR_WEBSITE_NAME", $FromName);

			$mail->setSender(array($MailFrom, $FromName));
			$mail->setSubject(JText::sprintf('SUBJECT_ORDER_NUMBER', $order->order_number));
			$mail->setBody($mail_body);
			$mail->IsHTML(true);
			$mail->addRecipient($order->email);

			$sent = $mail->Send();

			$this->setMessage(JText::_('COM_TAXIBOOKING_SEND_NOTES_SUCCESS'));
		}
		else {
			$this->setMessage(JText::_('COM_TAXIBOOKING_SEND_NOTES_FAILED'), 'error');
		}

		// Redirect back to the details screen.
		JRequest::setVar( 'layout', 'default' );
		$this->setRedirect(
			JRoute::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_item
				. $this->getRedirectToItemAppend($order_id), false
			)
		);
	}

	/**
        * Method to save the submitted ordering values for records via AJAX.
        *
        * @return  void
        *
        * @since   3.0
        */
	public function getCustomFieldHtmlAjax()
	{
		$json_result = array('error' => 0, 'msg' => '');
		$db = JFactory::getDBO();

		$field_id = JRequest::getInt('field_id', 0);

		if($field_id > 0)
		{
			$query = 'SELECT f.*'
				. ' FROM #__taxibooking_fields AS f'
				. ' WHERE f.id = '.(int)$field_id
			;
			$db->setQuery($query);
			$db->query();
			$row = $db->loadObject();

			if(!empty($row))
			{
				$title = $row->title;
				if($row->price > 0){
					$title .= '&nbsp;('.booking_helper::price_display((float)$row->price).')';
				}

				$html = '<div class="admin_list_item clearfix tr_col_from">
				<div class="admin_list_item_one"><label>'.$title.'</label></div>';

				$html .=    '<div class="admin_list_item_two">';
				if($row->field_type=='input'){
				    $additional_class = ($row->is_mandatory==1) ? ' required' : '';
				    $html .=    '<input class="inputbox extra'.$additional_class.'" type="text" value="" name="extras[pickup]['.$row->id.']" />';
				}
				elseif($row->field_type=='extra'){
				    $html .=    '<select class="select extra" name="extras[pickup]['.$row->id.']">';

				    $start = ($row->allow_zero==1) ? 0 : 1;
				    for($i = $start; $i <= $row->quantity; $i++){
					$html .= '<option value="'.$i.'">'.$i.'</option>';
				    }

				    $html .=    '</select>';
				}
				elseif($row->field_type=='optionlist'){
				    $html .=    '<select class="select extra" name="extras[pickup]['.$row->id.']">';

				    $field_options = unserialize($row->field_options);
				    for($i = 0; $i < count($field_options); $i++){
					$html .= '<option value="'.htmlspecialchars($field_options[$i]).'">'.$field_options[$i].'</option>';
				    }

				    $html .=    '</select>';
				}
				$html .=    '<a href="javascript:void(0);" class="remove_custom_field btn">'.JText::_('REMOVE').'</a></div>';
				$html .=    '</div>';

				$json_result['msg'] = $html;
			}
			else {
				$json_result['error'] = 1;
				$json_result['msg'] = JText::_('ORDER_EDIT_CUSTOM_FIELD_NOT_FOUND');
			}
		}
		else {
			$json_result['error'] = 1;
			$json_result['msg'] = JText::_('ORDER_EDIT_CUSTOM_FIELD_NOT_FOUND');
		}

		echo json_encode($json_result);
		exit();
	}

	/**
        * Method to save the submitted ordering values for records via AJAX.
        *
        * @return  void
        *
        * @since   3.0
        */
	public function getAvailableCarsAjax()
	{
		$json_result = array('error' => 0, 'msg' => '', 'selected_car' => 0, 'selected_car_price' => 0);
		$db = JFactory::getDBO();

		require_once (JPATH_COMPONENT_SITE.DS.'classes'.DS.'adminbooking.php');
		$orderClass = new TBAdminBooking();

		$editing_order_id = JRequest::getInt('id', 0);
		$booking_type = JRequest::getVar('booking_type', 'address');

		$order_data = new stdClass;
		$order_data->booking_type = $booking_type;

		$order_data->pickup_poi = JRequest::getInt('pickup_poi', 0);
		$order_data->dropoff_poi = JRequest::getInt('dropoff_poi', 0);

		$order_data->pickup_date = JRequest::getVar('pickup_date', '').' '.JRequest::getVar('pickup_hr', '').':'.JRequest::getVar('pickup_min', '');

		$order_data->route_id = JRequest::getInt('route_id', 0);

		$order_data->hourly_hr = JRequest::getInt('hourly_hr', 0);
		$order_data->hourly_min = JRequest::getInt('hourly_min', 0);

		$order_data->shuttle_pickup_poi = JRequest::getInt('shuttle_pickup_poi', 0);
		$order_data->shuttle_dropoff_poi = JRequest::getInt('shuttle_dropoff_poi', 0);
		$order_data->shuttle_pickup_date = JRequest::getVar('shuttle_pickup_date', 0);
		$order_data->shuttletime = JRequest::getVar('shuttletime', 0);
		$order_data->shuttle_passengers = JRequest::getInt('shuttle_passengers', 0);
		$order_data->shuttle_route_id = JRequest::getInt('shuttle_route_id', 0);

		$order_data->adultseats = JRequest::getInt('selpassengers', 0);
		$order_data->suitcases = JRequest::getInt('selluggage', 0);
		$order_data->infantseats = JRequest::getInt('selinfantseats', 0);
		$order_data->childseats = JRequest::getInt('selchildseats', 0);
		$order_data->boosterseats = JRequest::getInt('selboosterseats', 0);
		$order_data->returntrip = JRequest::getInt('returntrip', 0);
		$order_data->extras = JRequest::getVar('extras', array(), 'post', 'array');

		if($booking_type=='address')
		{
			if($order_data->pickup_poi>0){
				$poi_where = array('p.id = '.$db->Quote($order_data->pickup_poi));
				$poiObj = booking_helper::get_specific_place_details($poi_where);
				if($poiObj){
				    $order_data->pickup_address = $poiObj->title;
				    $order_data->pickup_coords = array($poiObj->lat, $poiObj->long);
				}
			}
			else {
				$order_data->pickup_address = JRequest::getVar('pickup_address', '');
				$order_data->pickup_coords = array(JRequest::getVar('pickup_lat', ''), JRequest::getVar('pickup_lng', ''));
			}

			if($order_data->dropoff_poi > 0){
				$poi_where = array('p.id = '.$db->Quote($order_data->dropoff_poi));
				$poiObj = booking_helper::get_specific_place_details($poi_where);
				if($poiObj){
				    $order_data->dropoff_address = $poiObj->title;
				    $order_data->dropoff_coords = array($poiObj->lat, $poiObj->long);
				}
			}
			else {
				$order_data->dropoff_address = JRequest::getVar('dropoff_address', '');
				$order_data->dropoff_coords = array(JRequest::getVar('dropoff_lat', ''), JRequest::getVar('dropoff_lng', ''));
			}

			$waypoints_post = JRequest::getVar('waypoints', array(), 'post', 'array');
			$waypoints_lat_post = JRequest::getVar('waypoints_lat', array(), 'post', 'array');
			$waypoints_lng_post = JRequest::getVar('waypoints_lng', array(), 'post', 'array');
			$waypoints_stop_duration_post = JRequest::getVar('waypoints_stop_duration', array(), 'post', 'array');

			$waypoints = $waypoints_lat = $waypoints_lng = $waypoints_stop_duration = $waypoint_coords = array();
			$total_waypoint_duration = 0;

			for($i=0;$i<count($waypoints_post);$i++)
			{
				if($waypoints_post[$i]!="" && $waypoints_lat_post[$i]!="" && $waypoints_lng_post[$i]!="")
				{
					$waypoints[] = $waypoints_post[$i];
					$waypoints_lat[] = $waypoints_lat_post[$i];
					$waypoints_lng[] = $waypoints_lng_post[$i];
					$waypoints_stop_duration[] = $waypoints_stop_duration_post[$i];

					$total_waypoint_duration += $waypoints_stop_duration_post[$i];

					$waypoint_coords[] = array($waypoints_lat_post[$i], $waypoints_lng_post[$i]);
				}
			}

			$order_data->waypoint_coords = $waypoint_coords;
			$order_data->total_waypoint_duration = $total_waypoint_duration;
		}
		elseif($booking_type=='offers')
		{
			$route_id = $order_data->route_id;
			$query = $db->getQuery(true);
			$query->select('route_from, route_to');
			$query->from('#__taxibooking_routes');
			$query->where('id = '.(int)$route_id);
			$db->setQuery((string)$query);
			$selected_route = $db->loadObject();

			if($selected_route)
			{
				$poi_where = array('p.id = '.$db->Quote($selected_route->route_from));
				$poiObj = booking_helper::get_specific_place_details($poi_where);
				if($poiObj){
					$order_data->pickup_poi = $poiObj->id;
					$order_data->pickup_coords = array($poiObj->lat, $poiObj->long);
				}

				$poi_where = array('p.id = '.$db->Quote($selected_route->route_to));
				$poiObj = booking_helper::get_specific_place_details($poi_where);
				if($poiObj){
					$order_data->dropoff_poi = $poiObj->id;
					$order_data->dropoff_coords = array($poiObj->lat, $poiObj->long);
				}
			}
		}

		$order_data->editing_order_id = $editing_order_id;
		$result = $orderClass->getAvailableCars($order_data);

		if($result['error']==1)
		{
			$json_result['error'] = 1;
			$json_result['msg'] = $result['msg'];
		}
		else {
			if(!empty($result['available_cars']))
			{
				$available_cars = $result['available_cars'];

				if($editing_order_id > 0){
					$editOrderObj = booking_helper::get_order_by_id($editing_order_id);
				}

				ob_start();
?>
<?php if(version_compare(JVERSION,'3.1.5','ge')) {?>
<table class="table table-striped no-more-tables">
<?php } else { ?>
<table class="adminlist">
<?php } ?>
	<thead>
		<tr>
			<th class="title" width="20%"><?php echo JText::_('TITLE');?></th>
			<th class="numeric" width="20%"><?php echo JText::_( 'IMAGE' ); ?></th>
			<th class="numeric" width="10%"><?php echo JText::_( 'PASSENGER_NO' );?></th>
			<th class="numeric" width="10%"><?php echo JText::_( 'SUITCASE_NO' );?></th>
			<th class="numeric" width="20%"><?php echo JText::_( 'PRICE' );?></th>
			<th class="numeric" width="20%"><?php echo JText::_( 'JSELECT' );?></th>
		</tr>
	</thead>
	<tbody>
		<?php
		$k = 0;
		for ($i=0, $n=count($available_cars); $i < $n; $i++) {
			$car = $available_cars[$i];
		?>
		<tr class="car_row <?php echo "row$k"; ?>" data-price="<?php echo $car->car_price ;?>">
			<td data-title="<?php echo JText::_('TITLE');?>" align="center">
				<span><?php echo htmlspecialchars($car->title, ENT_QUOTES, 'UTF-8'); ?></span>
			</td>
			<td data-title="<?php echo JText::_( 'IMAGE' ); ?>" class="numeric" align="center">
				<img src="<?php echo JURI::root().$car->image;?>" alt="<?php echo $car->title;?>" width="50" />
			</td>
			<td data-title="<?php echo JText::_('PASSENGER_NO');?>" class="numeric" align="center"><?php echo $car->passenger_no; ?></td>
			<td data-title="<?php echo JText::_('SUITCASE_NO');?>" class="numeric" align="center"><?php echo $car->suitcase_no; ?></td>
			<td data-title="<?php echo JText::_('PRICE');?>" class="numeric" align="center"><?php echo booking_helper::price_display((float)$car->car_price); ?></td>
			<td data-title="Assign" class="numeric" align="center">
				<?php
				if($editing_order_id > 0){
					if($editOrderObj->vehicletype==$car->id){
						$checked = ' checked="checked"';
						$json_result['selected_car'] = $car->id;
						$json_result['selected_car_price'] = $car->car_price;
					}
					else {
						$checked = '';
					}
				}
				else {
					$checked = '';
				}
				?>
				<input type="radio" class="assign_car" name="car_id" id="car_<?php echo $car->id;?>" value="<?php echo $car->id;?>"<?php echo $checked;?> />
			</td>
		</tr>
		<?php $k = 1 - $k; } ?>
	</tbody>
</table>

<input type="hidden" name="distance" id="distance" value="<?php echo !empty($result['additional_params']['distance']) ? $result['additional_params']['distance'] : 0;?>" />
<input type="hidden" name="duration_seconds" id="duration_seconds" value="<?php echo !empty($result['additional_params']['duration_seconds']) ? $result['additional_params']['duration_seconds'] : 0;?>" />
<input type="hidden" name="base_pickup_duration" id="base_pickup_duration" value="<?php echo !empty($result['additional_params']['base_pickup_duration']) ? $result['additional_params']['base_pickup_duration'] : 0;?>" />
<input type="hidden" name="dropoff_base_duration" id="dropoff_base_duration" value="<?php echo !empty($result['additional_params']['dropoff_base_duration']) ? $result['additional_params']['dropoff_base_duration'] : 0;?>" />

<?php

				$html = ob_get_contents();
				ob_end_clean();

				$json_result['msg'] = $html;
			}
			else {
				$json_result['error'] = 1;
				$json_result['msg'] = JText::_('ORDER_EDIT_ASSIGN_CAR_NOT_FOUND');
			}
		}

		echo json_encode($json_result);
		exit();
	}

	public function cancelFrontPaymentAjax()
	{
		$json_result = array('error' => 0, 'msg' => '');
		$db = JFactory::getDBO();

		$order_id = JRequest::getInt('id', 0);

		if($order_id > 0)
		{
			$orderObj = booking_helper::get_order_by_id($order_id);
			$paymentObj = booking_helper::get_payment_details($orderObj->payment);
			if($paymentObj){
				$payment_tablename = '#__taxibooking_payment_plg_' . $paymentObj->payment_element;

				$query = "DELETE FROM " . $db->quoteName($payment_tablename) . " WHERE order_id = ".(int)$order_id;
				$db->setQuery($query);
				$db->execute();

				// update payment in order table
				$query = "UPDATE " . $db->quoteName('#__taxibooking_orders')
				. " SET `payment` = NULL,"
				. " `cprice` = `cprice` - `flat_cost` - `percentage_cost`, "
				. " `flat_cost` = '0', `percentage_cost` = '0' WHERE id = ".(int)$order_id;
				$db->setQuery($query);
				$db->execute();

				$json_result['error'] = 0;
				$json_result['msg'] = JText::_('ORDER_EDIT_CANCEL_PAYMENT_SUCCESS');
			}
			else {
				$json_result['error'] = 1;
				$json_result['msg'] = JText::_('ORDER_EDIT_CANCEL_PAYMENT_ALREADY');
			}
		}
		else {
			$json_result['error'] = 1;
			$json_result['msg'] = 'Order not found!';
		}

		echo json_encode($json_result);
		exit();
	}

	public function changeStatusAjax()
	{
		$json_result = array('error' => 0, 'msg' => '');
		$app   = JFactory::getApplication();
		$db =  JFactory::getDBO();
		$date = JFactory::getDate();
		$elsettings 	= booking_helper::config();
		$siteOffset = booking_helper::set_tbtimezone();
		$dtnow = JFactory::getDate('now', $siteOffset);
		$lang =  JFactory::getLanguage();
		$lang_tag = $lang->getTag();

		$order_id = JRequest::getInt('id', 0);
		$new_status = JRequest::getInt('new_status', 0);

		if($order_id > 0)
		{
			$temp = new stdClass();
			$temp->id = $order_id;
			$temp->state = $new_status;
			$db->updateObject('#__taxibooking_orders', $temp, 'id');

			$row_queue = booking_helper::get_order_by_id($order_id);

			// language on which order was submitted, will be used at backend also
			$lang_tag = ($row_queue->language=='*') ? 'en-GB' : $row_queue->language;
			$lang->load('com_taxibooking', JPATH_ADMINISTRATOR, 'en-GB', true);
			$lang->load('com_taxibooking', JPATH_ADMINISTRATOR, $lang_tag, true);

			$SiteName = $app->getCfg('sitename');
			$MailFrom = $app->getCfg('mailfrom');
			$FromName = $app->getCfg('fromname');
			$FromName = JText::sprintf("NEW_ORDER_EMAIL_YOUR_WEBSITE_NAME", $FromName);

			if($row_queue)
			{
				if($new_status == -1) // Archived email
				{
					booking_helper::sendOrderArchiveEmail($row_queue, $elsettings);
				}
				else {
					if ($row_queue->state == 1) {
						$row_queue->order_status .= '<span style="color:green;">'.JText::_('NEW_ORDER_STATUS_ACCEPTED').'</span>';
					} elseif ($row_queue->state == 0) {
						$row_queue->order_status .= '<span style="color:red;">'.JText::_('NEW_ORDER_STATUS_REJECTED').'</span>';
					} elseif ($row_queue->state == -1) {
						$row_queue->order_status .= '<span style="color:yellow;">'.JText::_('NEW_ORDER_STATUS_ARCHIVED').'</span>';
					} elseif ($row_queue->state == -2) {
						$row_queue->order_status .= '<span style="color:green;">'.JText::_('NEW_ORDER_STATUS_WAITING').'</span>';
					}

					$payment_data = '';
					JPluginHelper::importPlugin('tbpayment');
					$_dispatcher = JDispatcher::getInstance();
					$_returnValues = $_dispatcher->trigger('plgTbOnShowOrderEmailsInvoice',array( $row_queue->id,$row_queue->payment));
					foreach ($_returnValues as $_returnValue) {
						if ($_returnValue !== null) {
							$payment_data .= $_returnValue;
						}
					}

					$converted_currency_label = '';
					$returnValues = $_dispatcher->trigger('plgTbonShowConvertedPriceLabel', array(&$row_queue->payment, $row_queue->cprice, &$converted_currency_label));

					// glean language specific header and footer info
					$header_info = $elsettings->header_info;
					$header_info = isset($header_info[$lang_tag]) ? changeEditorImageUrl($header_info[$lang_tag]) : '';

					$contact_info = $elsettings->contact_info;
					$contact_info = isset($contact_info[$lang_tag]) ? changeEditorImageUrl($contact_info[$lang_tag]) : '';

					if($row_queue->driver_id > 0){
						$driverObj = booking_helper::getDriverInfo($row_queue->driver_id);
					}

					// now get the email template
					ob_start();
					include( JPATH_COMPONENT_SITE.DS .'templates'.DS.'order_emails'.DS.'confirmation_email.tpl.php' );
					$mailbody = ob_get_contents();
					ob_end_clean();

					// Creating a pdf invoice if status is changed to Accepted
					$pdf_file = '';

					if ($row_queue->state == 1)
					{
						$header_info = $elsettings->header_info;
						$header_info_pdf = isset($header_info[$lang_tag]) ? changeEditorImagePath($header_info[$lang_tag]) : '';

						$contact_info = $elsettings->contact_info;
						$contact_info_pdf = isset($contact_info[$lang_tag]) ? changeEditorImagePath($contact_info[$lang_tag]) : '';

						ob_start();
						include( JPATH_COMPONENT_SITE.DS .'templates'.DS.'order_invoice.tpl.php' );
						$invoice_template = ob_get_contents();
						ob_end_clean();

						// require_once(dirname(dirname(__FILE__))."/classes/dompdf_lib.php");
						// $dompdf_lib = new Dompdf_lib;
						// $dompdf_lib->set_custom_paper('A4');
						// $pdfoutput = $dompdf_lib->convert_html_to_pdf($invoice_template, '', false);

						$invoice_title = ($elsettings->invoice_title!="") ? $elsettings->invoice_title."_".$order_id.".pdf" : "taxi_reservation_$order_id.pdf";

						$pdf_file = JPATH_COMPONENT . DS . "documents".DS.$invoice_title;
						// file_put_contents($pdf_file, $pdfoutput);

						// new pdf 
						// require(dirname(dirname(__FILE__))."/dompdf/autoload.inc.php");

						// use Dompdf\Dompdf;
						$dompdf = new Dompdf();

						$optionsPdf = $dompdf->getOptions();
						$optionsPdf->setIsHtml5ParserEnabled('true');
						$optionsPdf->isRemoteEnabled('true');
						$dompdf->setOptions($optionsPdf);
						$dompdf->loadHtml($invoice_template);

						// (Optional) Setup the paper size and orientation
						$dompdf->setPaper('A4', 'portrait');

						// Render the HTML as PDF
						$dompdf->render();

						// Output the generated PDF to Browser
						// $dompdf->stream();
						file_put_contents($pdf_file, $dompdf->output());

						// update order invoice title
						$query = 'UPDATE #__taxibooking_orders SET'
							.' invoice_title =' . $db->Quote($invoice_title)
							.' WHERE id = ' . (int)$order_id;
						$db->setQuery($query);
						$db->query();

						// confirm booked seats if this is shuttle booking
						if($row_queue->booking_type=='shuttle'){
							booking_helper::confirm_shuttle_booking($row_queue);
						}

						// send sms
						if((int)$elsettings->send_user_sms==1 && $row_queue->phone!="")
						{
							// get the order info
							$query = 'SELECT o.*, pm.title AS payment_method
								FROM #__taxibooking_orders AS o
								LEFT JOIN #__taxibooking_paymentmethods AS pm ON pm.id = o.payment
								WHERE o.id = ' . (int)$order_id;
							$db->setQuery($query);
							$db->query();
							$order_info = $db->loadObject();

							$response = booking_helper::send_order_sms($order_info, $elsettings);
							$app->enqueueMessage($response, 'notice');
						}
					}

					$mail =  JFactory::getMailer();
					$mail->setSender(array($MailFrom, $FromName));
					$mail->setSubject(JText::sprintf('BOOKING_STATUS_CHANGED_ORDER', $row_queue->order_number));
					$mail->setBody($mailbody);
					if($pdf_file!=""&&$elsettings->show_price==1&&strpos(strtolower($row_queue->custom_payment), 'jayride')===false)
					{
					    $mail->addAttachment($pdf_file);
					}
					$mail->IsHTML(true);
					$mail->addRecipient($row_queue->email);
					$sent = $mail->Send();

					// Remove car booking data if order status is changed to rejected
					if ($row_queue->state == 0){
						booking_helper::cancel_car_bookings($order_id);
					}

					// changed order status is not Accpeted so, remove shuttle booking
					if($row_queue->booking_type=='shuttle' && $row_queue->state!=1){
						booking_helper::remove_shuttle_booking($order_id);
					}
				}

				$json_result['error'] = 0;
				$json_result['msg'] = JText::_('COM_TAXIBOOKING_CHANGE_SUCCESS');
			}
			else {
				$json_result['error'] = 1;
				$json_result['msg'] = 'Order not found!';
			}
		}
		else {
			$json_result['error'] = 1;
			$json_result['msg'] = 'Order not found!';
		}

		echo json_encode($json_result);
		exit();
	}

	public function changeDriverAjax()
	{
		$json_result = array('error' => 0, 'msg' => '');
		$app   = JFactory::getApplication();
		$db =  JFactory::getDBO();
		$date = JFactory::getDate();
		$elsettings 	= booking_helper::config();
		$siteOffset = booking_helper::set_tbtimezone();
		$dtnow = JFactory::getDate('now', $siteOffset);
		$lang =  JFactory::getLanguage();
		$lang_tag = $lang->getTag();

		$order_id = JRequest::getInt('id', 0);
		$new_driver = JRequest::getInt('new_driver', 0);

		$this->sendDriverNotification($new_driver, $order_id);

		if($order_id > 0)
		{
			$temp = new stdClass();
			$temp->id = $order_id;
			$temp->driver_id = $new_driver;
			$db->updateObject('#__taxibooking_orders', $temp, 'id');

			$row_queue = booking_helper::get_order_by_id($order_id);

			// language on which order was submitted, will be used at backend also
			$lang_tag = ($row_queue->language=='*') ? 'en-GB' : $row_queue->language;
			$lang->load('com_taxibooking', JPATH_ADMINISTRATOR, 'en-GB', true);
			$lang->load('com_taxibooking', JPATH_ADMINISTRATOR, $lang_tag, true);

			if($row_queue)
			{
				if ($row_queue->state == 1) {
					$row_queue->order_status .= '<span style="color:green;">'.JText::_('NEW_ORDER_STATUS_ACCEPTED').'</span>';
				} elseif ($row_queue->state == 0) {
					$row_queue->order_status .= '<span style="color:red;">'.JText::_('NEW_ORDER_STATUS_REJECTED').'</span>';
				} elseif ($row_queue->state == -1) {
					$row_queue->order_status .= '<span style="color:yellow;">'.JText::_('NEW_ORDER_STATUS_ARCHIVED').'</span>';
				} elseif ($row_queue->state == -2) {
					$row_queue->order_status .= '<span style="color:green;">'.JText::_('NEW_ORDER_STATUS_WAITING').'</span>';
				}

				$payment_data = '';
				JPluginHelper::importPlugin('tbpayment');
				$_dispatcher = JDispatcher::getInstance();
				$_returnValues = $_dispatcher->trigger('plgTbOnShowOrderEmailsInvoice',array( $row_queue->id,$row_queue->payment));
				foreach ($_returnValues as $_returnValue) {
					if ($_returnValue !== null) {
						$payment_data .= $_returnValue;
					}
				}

				$converted_currency_label = '';
				$returnValues = $_dispatcher->trigger('plgTbonShowConvertedPriceLabel', array(&$row_queue->payment, $row_queue->cprice, &$converted_currency_label));

				// glean language specific header and footer info
				$header_info = $elsettings->header_info;
				$header_info = isset($header_info[$lang_tag]) ? changeEditorImageUrl($header_info[$lang_tag]) : '';

				$contact_info = $elsettings->contact_info;
				$contact_info = isset($contact_info[$lang_tag]) ? changeEditorImageUrl($contact_info[$lang_tag]) : '';

				if($row_queue->driver_id > 0){
					$driverObj = booking_helper::getDriverInfo($row_queue->driver_id);
				}

				// now get the email template
				ob_start();
				include( JPATH_COMPONENT_SITE.DS .'templates'.DS.'order_emails'.DS.'confirmation_email.tpl.php' );
				$mailbody = ob_get_contents();
				ob_end_clean();

				// Creating a pdf invoice if status is changed to Accepted
				$pdf_file = '';
				if ($row_queue->state == 1)
				{
					$header_info = $elsettings->header_info;
					$header_info_pdf = isset($header_info[$lang_tag]) ? changeEditorImagePath($header_info[$lang_tag]) : '';

					$contact_info = $elsettings->contact_info;
					$contact_info_pdf = isset($contact_info[$lang_tag]) ? changeEditorImagePath($contact_info[$lang_tag]) : '';

					ob_start();
					include( JPATH_COMPONENT_SITE.DS .'templates'.DS.'order_invoice.tpl.php' );
					$invoice_template = ob_get_contents();
					ob_end_clean();

					// require_once(dirname(dirname(__FILE__))."/classes/dompdf_lib.php");
					// $dompdf_lib = new Dompdf_lib;
					// $dompdf_lib->set_custom_paper('A4');
					// $pdfoutput = $dompdf_lib->convert_html_to_pdf($invoice_template, '', false);

					$invoice_title = ($elsettings->invoice_title!="") ? $elsettings->invoice_title."_".$order_id.".pdf" : "taxi_reservation_$order_id.pdf";

					$pdf_file = JPATH_COMPONENT . DS . "documents".DS.$invoice_title;
					// file_put_contents($pdf_file, $pdfoutput);

					// fix pdf -> switch to upgraded version of dompdf
					$dompdf = new Dompdf();

					$optionsPdf = $dompdf->getOptions();
					$optionsPdf->setIsHtml5ParserEnabled('true');
					$optionsPdf->isRemoteEnabled('true');
					$dompdf->setOptions($optionsPdf);
					$dompdf->loadHtml($invoice_template);

					// (Optional) Setup the paper size and orientation
					$dompdf->setPaper('A4', 'portrait');

					// Render the HTML as PDF
					$dompdf->render();

					file_put_contents($pdf_file, $dompdf->output());

					// update order invoice title
					$query = 'UPDATE #__taxibooking_orders SET'
						.' invoice_title =' . $db->Quote($invoice_title)
						.' WHERE id = ' . (int)$order_id;
					$db->setQuery($query);
					$db->query();

				}

				$SiteName = $app->getCfg('sitename');
				$MailFrom = $app->getCfg('mailfrom');
				$FromName = $app->getCfg('fromname');
				$FromName = JText::sprintf("NEW_ORDER_EMAIL_YOUR_WEBSITE_NAME", $FromName);

				$mail =  JFactory::getMailer();
				$mail->setSender(array($MailFrom, $FromName));
				$mail->setSubject(JText::sprintf('BOOKING_STATUS_CHANGED_ORDER', $row_queue->order_number));
				$mail->setBody($mailbody);
				if($pdf_file!=""&&$elsettings->show_price==1&&strpos(strtolower($row_queue->custom_payment), 'jayride')===false)
				{
				    $mail->addAttachment($pdf_file);
				}
				$mail->IsHTML(true);
				$mail->addRecipient($row_queue->email);
				$sent = $mail->Send();

				// Sending mail to driver if driver is selected
				if ($new_driver > 0)
				{
					$query = 'SELECT name, email FROM #__users WHERE id = ' . $db->Quote($new_driver);
					$db->setQuery($query);
					$db->query();
					$driver_obj = $db->loadObject();
					if($driver_obj)
					{
					    $driver_name = $driver_obj->name;
					    $driver_email = $driver_obj->email;

					    $receiver='driver';
						// now get the email template
						ob_start();
						include( JPATH_COMPONENT_SITE.DS .'templates'.DS.'order_emails'.DS.'admin_email.tpl.php' );
						$mailbody = ob_get_contents();
						ob_end_clean();

					    $mail->ClearAllRecipients();
					    $mail->ClearAttachments();
					    $mail->setSender(array($MailFrom, $FromName));
					    $mail->setSubject(JText::sprintf('DRIVER_BOOKING_NOTIFICATION', $driver_email) .' '.JText::sprintf('BOOKING_STATUS_CHANGED_ORDER', $row_queue->order_number));
					    $mail->setBody($mailbody);
					    if($pdf_file!=""&&$elsettings->show_price==1&&strpos(strtolower($row_queue->custom_payment), 'jayride')===false)
					    {
						$mail->addAttachment($pdf_file);
					    }
					    $mail->IsHTML(true);
					    $mail->addRecipient($driver_email);
					    $sent = $mail->Send();
					}
				}

				$json_result['error'] = 0;
				$json_result['msg'] = JText::_('COM_TAXIBOOKING_CHANGE_SUCCESS');
			}
			else {
				$json_result['error'] = 1;
				$json_result['msg'] = 'Order not found!';
			}
		}
		else {
			$json_result['error'] = 1;
			$json_result['msg'] = 'Order not found!';
		}

		echo json_encode($json_result);
		exit();
	}

	private function sendDriverNotification($driver_id, $order_id)
	{
		$deviceTokens = $this->getDeviceTokens($driver_id);
		$order = $this->getOrderById($order_id);

		$subject = 'New order from Kabsky on ' . date('Y-m-d h:m', $order->datetime1);
		$message .= 'Pickup: ' . $order->begin;
		$message .= ' - Price: ' . $order->cprice * 0.80; // take 20% off
		// $message .= ' - Date: ' . date('Y-m-d h:m', $order->datetime1);
		// echo $message;
		try {
			foreach ($deviceTokens as $token) {
				$this->sendPush($token, $subject, $message, $driver_id, $order_id);
			}
		} catch(\Exception $e) {
			echo $e->getMessage();
		}
	}

	private function getDeviceTokens($id) 
	{
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);

		$query->select('token');
		$query->from($db->quoteName('#__user_tokens'));
		$query->where($db->quoteName('user_id')." = ".$db->quote($id));

		$db->setQuery($query);

		$result = $db->loadAssocList();
		$cleanResults = [];
		foreach ($result as $key => $value) {
			$cleanResults[] = $value['token'];
		}
		return $cleanResults;
	}

	private function getOrderById($id) 
	{
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);

		// $query->select('id', 'names', 'cprice', 'datetime1', 'begin', 'end');
		$query->select($db->quoteName(array('id', 'names', 'cprice', 'datetime1', 'begin', 'end')));
		$query->from($db->quoteName('#__taxibooking_orders'));
		$query->where($db->quoteName('id')." = ".$db->quote($id));

		$db->setQuery($query);

		return $db->loadObject();
	
	}

	function sendPush($token, $subject, $message, $uid, $order_id)
	{
		$content = array(
			"en" => $message
		);
		
		$headings = array(
			"en" => $subject
		);

		$buttons = array(
			["id" => "accepted",
			"text" => "Accept job"],
			["id" => "refused",
			"text" => "Reject job"],
		);

		$data = array(
			"orderId" => $order_id
		);
		
		$payload = array(
			'app_id' => "f9a18580-b00f-40fb-aab0-e8f449c01e7b",
			'include_player_ids' => array($token),
			'headings' => $headings,
			'contents' => $content,
			'data' => $data,
			"buttons" => $buttons
		);
		
		$fields = json_encode($payload);

		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, "https://onesignal.com/api/v1/notifications");
		curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json; charset=utf-8'));
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
		curl_setopt($ch, CURLOPT_HEADER, FALSE);
		curl_setopt($ch, CURLOPT_POST, TRUE);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $fields);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);

		$result = curl_exec($ch);
		$res = json_decode($result);

		if ($res->data->status == 'ok') {
			return true;
		}
		return false;
	}
	
}
