<?php
/**
 * @package     Joomla.Administrator
 * @subpackage  com_taxibooking
 *
 * @copyright   Copyright (C) 2005 - 2012 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined( '_JEXEC' ) or die( 'Restricted access' );

jimport('joomla.application.component.controller');

/**
 * Component Item Controller
 */
class TaxiBookingControllerSearchurls extends TaxiBookingController
{
	/**
	 * @var		string	The prefix to use with controller messages.
	 * @since	1.6
	 */
	protected $text_prefix = 'COM_TAXIBOOKING';
	/**
	 * The context for storing internal data, e.g. record.
	 *
	 * @var    string
	 * @since  12.2
	 */
	protected $context;
	/**
	 * The URL option for the component.
	 *
	 * @var    string
	 * @since  12.2
	 */
	protected $option;

	/**
	 * The URL view item variable.
	 *
	 * @var    string
	 * @since  12.2
	 */
	protected $view_item;

	/**
	 * The URL view list variable.
	 *
	 * @var    string
	 * @since  12.2
	 */
	protected $view_list;
	/**
	 * Constructor
	 *
	 * @since 1.0
	 */
	function __construct()
	{
		parent::__construct();
		
		$this->context = 'searchurl';
		// Guess the option as com_NameOfController
		if (empty($this->option))
		{
			$this->option = 'com_taxibooking';
		}
		// Guess the item view as the context.
		if (empty($this->view_item))
		{
			$this->view_item = 'searchurl';
		}
		// Guess the list view as the plural of the item view.
		if (empty($this->view_list))
		{
			$this->view_list = 'searchurls';
		}
		
		// Define standard task mappings.
		$this->registerTask( 'apply', 			'save' );
		$this->registerTask( 'save2copy', 		'save' );
		
		// Value = 0
		$this->registerTask('unpublish', 'publish');
		// Value = 2
		$this->registerTask('archive', 'publish');
		// Value = -2
		$this->registerTask('trash', 'publish');
		// Value = -3
		$this->registerTask('orderup', 'reorder');
		$this->registerTask('orderdown', 'reorder');
	}
	
	/**
	 * Method to add a new record.
	 *
	 * @return  mixed  True if the record can be added, a error object if not.
	 *
	 * @since   12.2
	 */
	public function add()
	{
		$app = JFactory::getApplication();
		$context = "$this->option.edit.$this->context";

		// Access check.
		if (!$this->allowAdd())
		{
			// Set the internal error and also the redirect error.
			$this->setError(JText::_('JLIB_APPLICATION_ERROR_CREATE_RECORD_NOT_PERMITTED'));
			$this->setMessage($this->getError(), 'error');

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_list
					. $this->getRedirectToListAppend(), false
				)
			);

			return false;
		}

		// Clear the record edit information from the session.
		$app->setUserState($context . '.data', null);

		// Redirect to the edit screen.
		$this->setRedirect(
			JRoute::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_item
				. $this->getRedirectToItemAppend(), false
			)
		);

		return true;
	}

	/**
	 * Method to check if you can add a new record.
	 *
	 * Extended classes can override this if necessary.
	 *
	 * @param   array  $data  An array of input data.
	 *
	 * @return  boolean
	 *
	 * @since   12.2
	 */
	protected function allowAdd($data = array())
	{
		$user = JFactory::getUser();
		return ($user->authorise('core.create', $this->option));
	}

	/**
	 * Method to cancel an edit.
	 *
	 * @param   string  $key  The name of the primary key of the URL variable.
	 *
	 * @return  boolean  True if access level checks pass, false otherwise.
	 *
	 * @since   12.2
	 */
	public function cancel()
	{
		JSession::checkToken() or jexit(JText::_('JINVALID_TOKEN'));

		$app = JFactory::getApplication();
		$model = $this->getModel('searchurl');
		$table = $model->getTable();
		$checkin = property_exists($table, 'checked_out');
		$context = "$this->option.edit.$this->context";

		if (empty($key))
		{
			$key = $table->getKeyName();
		}

		$recordId = $app->input->getInt($key);

		// Attempt to check-in the current record.
		if ($recordId)
		{
			if ($checkin)
			{
				if ($model->checkin($recordId) === false)
				{
					// Check-in failed, go back to the record and display a notice.
					$this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_CHECKIN_FAILED', $model->getError()));
					$this->setMessage($this->getError(), 'error');

					$this->setRedirect(
						JRoute::_(
							'index.php?option=' . $this->option . '&view=' . $this->view_item
							. $this->getRedirectToItemAppend($recordId, $key), false
						)
					);

					return false;
				}
			}
		}

		// Clean the session data and redirect.
		$this->releaseEditId($context, $recordId);
		$app->setUserState($context . '.data', null);

		$this->setRedirect(
			JRoute::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list
				. $this->getRedirectToListAppend(), false
			)
		);

		return true;
	}
	/**
	 * Method to check if you can add a new record.
	 *
	 * Extended classes can override this if necessary.
	 *
	 * @param   array   $data  An array of input data.
	 * @param   string  $key   The name of the key for the primary key; default is id.
	 *
	 * @return  boolean
	 *
	 * @since   12.2
	 */
	protected function allowEdit($data = array(), $key = 'id')
	{
		return JFactory::getUser()->authorise('core.edit', 'com_taxibooking');
	}
	
	/**
	 * Logic to create the view for edit order
	 *
	 * @access public
	 * @return void
	 * @since 1.0
	 */
	public function edit($key = null, $urlVar = null)
	{
		$app   = JFactory::getApplication();
		$model = $this->getModel('searchurl');
		$table = $model->getTable();
		$cid	= JRequest::getVar( 'cid', array(0), 'get', 'array' );
		$context = "$this->option.edit.$this->context";
		
		// Determine the name of the primary key for the data.
		if (empty($key))
		{
			$key = $table->getKeyName();
		}

		// To avoid data collisions the urlVar may be different from the primary key.
		if (empty($urlVar))
		{
			$urlVar = $key;
		}

		// Get the previous record id (if any) and the current record id.
		$recordId = (int) (count($cid) ? $cid[0] : $app->input->getInt($urlVar));
		$checkin = property_exists($table, 'checked_out');

		// Access check.
		if (!$this->allowEdit(array($key => $recordId), $key))
		{
			$this->setError(JText::_('JLIB_APPLICATION_ERROR_EDIT_NOT_PERMITTED'));
			$this->setMessage($this->getError(), 'error');

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_list
					. $this->getRedirectToListAppend(), false
				)
			);

			return false;
		}

		// Attempt to check-out the new record for editing and redirect.
		if ($checkin && !$model->checkout($recordId))
		{
			// Check-out failed, display a notice but allow the user to see the record.
			$this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_CHECKOUT_FAILED', $model->getError()));
			$this->setMessage($this->getError(), 'error');

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_item
					. $this->getRedirectToItemAppend($recordId, $urlVar), false
				)
			);

			return false;
		}
		else
		{
			// Check-out succeeded, push the new record id into the session.
			$this->holdEditId($context, $recordId);
			$app->setUserState($context . '.data', null);

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_item
					. $this->getRedirectToItemAppend($recordId, $urlVar), false
				)
			);

			return true;
		}
	}
	
        /**
	 * Logic to save an order
	 *
	 * @access public
	 * @return void
	 * @since 1.0
	 */
	public function save($key = null, $urlVar = null)
	{
		// Check for request forgeries.
		JSession::checkToken() or jexit(JText::_('JINVALID_TOKEN'));

		$app   = JFactory::getApplication();
		$db =  JFactory::getDBO();
		$date = JFactory::getDate();
		$model = $this->getModel('searchurl');
		$table = $model->getTable();
		$post = JRequest::get( 'post' );
		$post['text'] = JRequest::getVar( 'text', '', 'post', 'string', JREQUEST_ALLOWRAW );
		$checkin = property_exists($table, 'checked_out');
		$task = $this->getTask();
		$elsettings 	= booking_helper::config();
		$lang =  JFactory::getLanguage();
		$lang_tag = $lang->getTag();

		// Determine the name of the primary key for the data.
		if (empty($key))
		{
			$key = $table->getKeyName();
		}

		// To avoid data collisions the urlVar may be different from the primary key.
		if (empty($urlVar))
		{
			$urlVar = $key;
		}
		
		if($post['anchor_text']==''){
			$post['anchor_text'] = $post['title'];
		}
		
		// generate short URL
		$post['anchor_text_html'] = '<a href="'.$post['generated_url'].'" target="blank">'.$post['anchor_text'].'</a>';
		
		$params = JComponentHelper::getParams('com_taxibooking');
		$url_shortner_api_key = $params->get('url_shortner_api_key', '');
		if($url_shortner_api_key!="")
		{
			require_once (JPATH_COMPONENT_SITE.DS.'classes'.DS.'google_url.php');
			$api = new GoogleURL();
			$longUrl = $post['generated_url'];
			$postData = array('longUrl' => $longUrl, 'key' => $url_shortner_api_key);
			$info = $api->httpsPost($postData);
			
			if(!empty($info->error)){
				$app->enqueueMessage('Google URL Shortener API error code - '.$info->error->code.', message - '.$info->error->message.', reason - '.$info->error->errors[0]->reason, 'warning');
			}
			else {
				$post['short_url'] = $info->id;
				$post['anchor_text_html'] = '<a href="'.$post['short_url'].'" target="blank">'.$post['anchor_text'].'</a>';
			}
		}
		
		$recordId = JRequest::getInt($urlVar);
		
		if ( $model->save($post) ) {
			
			// Save succeeded, so check-in the record.
			if ($checkin && $model->checkin($recordId) === false)
			{
				// Save the data in the session.
				$app->setUserState($this->context . '.data', $post);
	
				// Check-in failed, so go back to the record and display a notice.
				$this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_CHECKIN_FAILED', $model->getError()));
				$this->setMessage($this->getError(), 'error');
	
				$this->setRedirect(
					JRoute::_(
						'index.php?option=' . $this->option . '&view=' . $this->view_item
						. $this->getRedirectToItemAppend($recordId, $urlVar), false
					)
				);
	
				return false;
			}
			
			$this->setMessage(JText::_('COM_TAXIBOOKING_SAVE_SUCCESS'));
			
			// Redirect the user and adjust session state based on the chosen task.
			switch ($task)
			{
				case 'apply':
					// Set the record data in the session.
					$recordId = $model->getState($this->context . '.id');
					$this->holdEditId($this->context, $recordId);
					$app->setUserState($this->context . '.data', null);
					$model->checkout($recordId);
	
					// Redirect back to the edit screen.
					$this->setRedirect(
						JRoute::_(
							'index.php?option=' . $this->option . '&view=' . $this->view_item
							. $this->getRedirectToItemAppend($recordId, $urlVar), false
						)
					);
					break;
	
				case 'save2new':
					// Clear the record id and data from the session.
					$this->releaseEditId($this->context, $recordId);
					$app->setUserState($this->context . '.data', null);
	
					// Redirect back to the edit screen.
					$this->setRedirect(
						JRoute::_(
							'index.php?option=' . $this->option . '&view=' . $this->view_item
							. $this->getRedirectToItemAppend(null, $urlVar), false
						)
					);
					break;
	
				default:
					// Clear the record id and data from the session.
					$this->releaseEditId($this->context, $recordId);
					$app->setUserState($this->context . '.data', null);
	
					// Redirect to the list screen.
					$this->setRedirect(
						JRoute::_(
							'index.php?option=' . $this->option . '&view=' . $this->view_list
							. $this->getRedirectToListAppend(), false
						)
					);
					break;
			}
			
			return true;

		} else {
			// Redirect back to the edit screen.
			$this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_SAVE_FAILED', $model->getError()));
			$this->setMessage($this->getError(), 'error');

			$this->setRedirect(
				JRoute::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_item
					. $this->getRedirectToItemAppend($recordId, $urlVar), false
				)
			);

			return false;
		}
	}
	
	/**
	 * Method to publish a list of items
	 *
	 * @return  void
	 *
	 * @since   12.2
	 */
	public function publish()
	{
		// Check for request forgeries
		JSession::checkToken() or die(JText::_('JINVALID_TOKEN'));

		// Get items to publish from the request.
		$cid	= JRequest::getVar( 'cid', array(), 'post', 'array' );
		$data = array('publish' => 1, 'unpublish' => 0, 'archive' => 2, 'trash' => -2, 'report' => -3);
		$task = $this->getTask();
		$value = JArrayHelper::getValue($data, $task, 0, 'int');

		if (empty($cid))
		{
			JLog::add(JText::_($this->text_prefix . '_NO_ITEM_SELECTED'), JLog::WARNING, 'jerror');
		}
		else
		{
			// Get the model.
			$model = $this->getModel('searchurl');

			// Make sure the item ids are integers
			JArrayHelper::toInteger($cid);

			// Publish the items.
			try
			{
				$model->publish($cid, $value);

				if ($value == 1)
				{
					$ntext = $this->text_prefix . '_N_ITEMS_PUBLISHED';
				}
				elseif ($value == 0)
				{
					$ntext = $this->text_prefix . '_N_ITEMS_UNPUBLISHED';
				}
				elseif ($value == 2)
				{
					$ntext = $this->text_prefix . '_N_ITEMS_ARCHIVED';
				}
				else
				{
					$ntext = $this->text_prefix . '_N_ITEMS_TRASHED';
				}
				$this->setMessage(JText::plural($ntext, count($cid)));
			}
			catch (Exception $e)
			{
				$this->setMessage(JText::_('JLIB_DATABASE_ERROR_ANCESTOR_NODES_LOWER_STATE'), 'error');
			}
		}
		
		$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false));
	}
	
	/**
	 * Changes the order of one or more records.
	 *
	 * @return  boolean  True on success
	 *
	 * @since   12.2
	 */
	public function reorder()
	{
		// Check for request forgeries.
		JSession::checkToken() or jexit(JText::_('JINVALID_TOKEN'));

		$ids	= JRequest::getVar( 'cid', array(), 'post', 'array' );
		$inc = ($this->getTask() == 'orderup') ? -1 : 1;

		$model = $this->getModel('searchurl');
		$return = $model->reorder($ids, $inc);
		if ($return === false)
		{
			// Reorder failed.
			$message = JText::sprintf('JLIB_APPLICATION_ERROR_REORDER_FAILED', $model->getError());
			$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false), $message, 'error');
			return false;
		}
		else
		{
			// Reorder succeeded.
			$message = JText::_('JLIB_APPLICATION_SUCCESS_ITEM_REORDERED');
			$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false), $message);
			return true;
		}
	}

	/**
	 * Method to save the submitted ordering values for records.
	 *
	 * @return  boolean  True on success
	 *
	 * @since   12.2
	 */
	public function saveorder()
	{
		// Check for request forgeries.
		JSession::checkToken() or jexit(JText::_('JINVALID_TOKEN'));

		// Get the input
		$pks	= JRequest::getVar( 'cid', array(), 'post', 'array' );
		$order	= JRequest::getVar( 'order', array(), 'post', 'array' );

		// Sanitize the input
		JArrayHelper::toInteger($pks);
		JArrayHelper::toInteger($order);

		// Get the model
		$model = $this->getModel('searchurl');

		// Save the ordering
		$return = $model->saveorder($pks, $order);

		if ($return === false)
		{
			// Reorder failed
			$message = JText::sprintf('JLIB_APPLICATION_ERROR_REORDER_FAILED', $model->getError());
			$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false), $message, 'error');
			return false;
		}
		else
		{
			// Reorder succeeded.
			$this->setMessage(JText::_('JLIB_APPLICATION_SUCCESS_ORDERING_SAVED'));
			$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false));
			return true;
		}
	}

	/**
	 * Logic to delete items
	 *
	 * @access public
	 * @return void
	 * @since 1.0
	 */
	public function remove()
	{
		// Check for request forgeries
		JSession::checkToken() or die(JText::_('JINVALID_TOKEN'));

		// Get items to remove from the request.
		$cid		= JRequest::getVar( 'cid', array(), 'post', 'array' );

		if (!is_array($cid) || count($cid) < 1)
		{
			JLog::add(JText::_($this->text_prefix . '_NO_ITEM_SELECTED'), JLog::WARNING, 'jerror');
		}
		else
		{
			// Get the model.
			$model = $this->getModel('searchurl');

			// Make sure the item ids are integers
			jimport('joomla.utilities.arrayhelper');
			JArrayHelper::toInteger($cid);

			// Remove the items.
			if ($model->delete($cid))
			{
				$this->setMessage(JText::plural($this->text_prefix . '_N_ITEMS_DELETED', count($cid)));
			}
			else
			{
				$this->setMessage($model->getError());
			}
		}
		
		$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false));
	}
	
	/**
	 * Check in of one or more records.
	 *
	 * @return  boolean  True on success
	 *
	 * @since   12.2
	 */
	public function checkin()
	{
		// Check for request forgeries.
		JSession::checkToken() or jexit(JText::_('JINVALID_TOKEN'));

		$ids		= JRequest::getVar( 'cid', array(), 'post', 'array' );

		$model = $this->getModel('searchurl');
		$return = $model->checkin($ids);
		if ($return === false)
		{
			// Checkin failed.
			$message = JText::sprintf('JLIB_APPLICATION_ERROR_CHECKIN_FAILED', $model->getError());
			$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false), $message, 'error');
			return false;
		}
		else
		{
			// Checkin succeeded.
			$message = JText::plural($this->text_prefix . '_N_ITEMS_CHECKED_IN', count($ids));
			$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false), $message);
			return true;
		}
	}
	
	/**
	 * Method to clone an existing module.
	 * @since	1.6
	 */
	public function duplicate()
	{
		// Check for request forgeries
		JRequest::checkToken() or jexit( 'Invalid Token' );

		// Initialise variables.
		$pks = JRequest::getVar('cid', array(), 'post', 'array');
		JArrayHelper::toInteger($pks);

		try {
			if (empty($pks)) {
				throw new Exception(JText::_($this->text_prefix . '_NO_ITEM_SELECTED'));
			}
			$model = $this->getModel('searchurl');
			$model->duplicate($pks);
			$this->setMessage(JText::plural('N_ITEMS_DUPLICATED', count($pks)));
		}
		catch (Exception $e) {
			JError::raiseWarning(500, $e->getMessage());
		}

		$this->setRedirect(JRoute::_('index.php?option=' . $this->option . '&view=' . $this->view_list, false));
		return true;
	}
	
	/**
        * Method to save the submitted ordering values for records via AJAX.
        *
        * @return  void
        *
        * @since   3.0
        */
	public function saveOrderAjax()
	{
		// Get the input
		$pks = JFactory::getApplication()->input->post->get('cid', array(), 'array');
		$order = JFactory::getApplication()->input->post->get('order', array(), 'array');
    
		// Sanitize the input
		JArrayHelper::toInteger($pks);
		JArrayHelper::toInteger($order);
    
		// Get the model
		$model = $this->getModel('searchurl');
    
		// Save the ordering
		$return = $model->saveorder($pks, $order);
    
		if ($return)
		{
			echo "1";
		}
    
		// Close the application
		JFactory::getApplication()->close();
	}
	
	/**
	* Get extras according to the user's choice Pickup and dropoff
	*
	* @access	public
	*/
	public function getExtraListAjax()
	{
		$result = array('error' => 0, 'msg' => '');
		$db = JFactory::getDBO();
		$lang =  JFactory::getLanguage();
		$lang_tag = JRequest::getVar('lang', '*');
		
		$id = JRequest::getInt('id', 0);
		$poi_id = JRequest::getInt('poiid', 0);
		$route_from = JRequest::getInt('route_from', 0);
		$route_to = JRequest::getInt('route_to', 0);
		$type = JRequest::getVar('type', 'pickup');
		$scope = JRequest::getVar('scope', 'searchurl');
		
		// get item details if id is present
		$selected_extras = array();
		if($id>0){
			$query = $db->getQuery(true);
			$query->select('*');
			$query->from('#__taxibooking_searchurls');
			$query->where('id = '.(int)$id);
			$db->setQuery((string)$query);
			$urlObj = $db->loadObject();
			
			if($urlObj){
				$params = unserialize($urlObj->url_params);
				$selected_extras = !empty($params['extras']) ? $params['extras'] : array();
			}
		}
		
		$where = array();
		$where[] = 'f.published = 1';
		
		if($lang_tag!="*"){
			$where[] = '(f.language = "*" OR f.language = "'.$lang_tag.'")';
		}
		
		switch($type){
		    
		    case 'pickup':
			if($poi_id > 0)
			{
			    $poi_obj = booking_helper::point_details($poi_id);
			    
			    $pickup_poicat = $poi_obj->catid;
			    $where[] = "(f.catid='$pickup_poicat' OR f.catid LIKE '$pickup_poicat,%' OR f.catid LIKE '%,$pickup_poicat' OR f.catid LIKE '%,$pickup_poicat,%')";
			    $where[] = 'f.show_on_pickup = 1';
			    $where[] = 'f.show_on_user_details = 0';
			    $where[] = 'f.show_on_shuttles = 0';
			}
			break;
		    
		    case 'dropoff':
			if($poi_id > 0)
			{
			    $poi_obj = booking_helper::point_details($poi_id);
			    
			    $dropoff_poicat = $poi_obj->catid;
			    $where[] = "(f.catid='$dropoff_poicat' OR f.catid LIKE '$dropoff_poicat,%' OR f.catid LIKE '%,$dropoff_poicat' OR f.catid LIKE '%,$dropoff_poicat,%')";
			    $where[] = 'f.show_on_dropoff = 1';
			    $where[] = 'f.show_on_user_details = 0';
			    $where[] = 'f.show_on_shuttles = 0';
			}
			break;
		    
		    case 'route_pickup':
			if($poi_id > 0)  // customer choose special route
			{
			    $poi_obj = booking_helper::point_details($poi_id);
			    $pickup_poicat = $poi_obj->catid;
			    
			    $where[] = "(f.catid='$pickup_poicat' OR f.catid LIKE '$pickup_poicat,%' OR f.catid LIKE '%,$pickup_poicat' OR f.catid LIKE '%,$pickup_poicat,%')";
			    $where[] = 'f.show_on_pickup = 1';
			    $where[] = 'f.show_on_user_details = 0';
			    $where[] = 'f.show_on_shuttles = 0';
			}
			break;
		    
		    case 'route_dropoff':
			if($poi_id > 0)
			{
			    $poi_obj = booking_helper::point_details($poi_id);
			    $dropoff_poicat = $poi_obj->catid;
			    
			    $where[] = "(f.catid='$dropoff_poicat' OR f.catid LIKE '$dropoff_poicat,%' OR f.catid LIKE '%,$dropoff_poicat' OR f.catid LIKE '%,$dropoff_poicat,%')";
			    $where[] = 'f.show_on_dropoff = 1';
			    $where[] = 'f.show_on_user_details = 0';
			    $where[] = 'f.show_on_shuttles = 0';
			}
			break;
		    
			case 'hourly_hire':
			
				$where[] = 'f.show_on_hourly = 1';
				$where[] = 'f.show_on_user_details = 0';
				break;
		
			case 'user_details':
			
				$where[] = 'f.show_on_user_details = 1';
				break;
		
			case 'address_pickup':
			
				$where[] = 'f.show_on_pickup = 1';
				$where[] = 'f.show_on_address = 1';
				$where[] = 'f.show_on_user_details = 0';
				$where[] = 'f.show_on_shuttles = 0';
				$type = 'pickup';
				break;
			
			case 'address_dropoff':
			
				$where[] = 'f.show_on_dropoff = 1';
				$where[] = 'f.show_on_address = 1';
				$where[] = 'f.show_on_user_details = 0';
				$where[] = 'f.show_on_shuttles = 0';
				$type = 'dropoff';
				break;
			
			default:
				break;
		}
		
		$where = ( count($where) ? ' WHERE ' . implode(' AND ', $where) : '' );
		$query = 'SELECT f.* '
			. ' FROM #__taxibooking_fields AS f'
			. $where
			. ' ORDER BY f.ordering ASC'
		;
		
		$db->setQuery($query);
		$db->query();
		$rows = $db->loadObjectList();
		
		$html = '';
		
		if(!empty($rows))
		{
			foreach($rows as $row)
			{
				if($scope=='orderedit'){
					$html .= '<div class="admin_list_item clearfix tr_col_from inputWrap">
						<div class="admin_list_item_one">
						    <label>'.$row->title.'</label>
						</div>
						<div class="admin_list_item_two">';
				}
				else {
					$html .= '<div class="inputWrap clearfix">
						<label>'.$row->title.'</label>';
				}
				
				$sel_value = '';	
				if(!empty($selected_extras[$type])){
					$sel_value = !empty($selected_extras[$type][$row->id]) ? $selected_extras[$type][$row->id] : '';
				}
			
				if($row->field_type=='input'){
				    $additional_class = '';
				    $html .= '<input class="inputbox extra'.$additional_class.'" type="text" value="'.$sel_value.'" name="extras['.$type.']['.$row->id.']" />';
				}
				elseif($row->field_type=='textarea'){
					$additional_class = '';
					$html .= '<textarea name="extras['.$type.']['.$row->id.']" class="inputbox extra'.$additional_class.'">'.$sel_value.'</textarea>';
				}
				elseif($row->field_type=='extra'){
				    $html .=    '<select class="inputbox select extra" name="extras['.$type.']['.$row->id.']">';
				    $html .= '<option value="">'.JText::_('JSELECT').'</option>';
				    $start = ($row->allow_zero==1) ? 0 : 1;
				    for($i = $start; $i <= $row->quantity; $i++){
					$selected = ($sel_value==$i) ? ' selected="selected"' : '';
					$html .= '<option value="'.$i.'"'.$selected.'>'.$i.'</option>';
				    }
				    
				    $html .=    '</select>';
				}
				elseif($row->field_type=='optionlist'){
				    $html .=    '<select class="inputbox select extra" name="extras['.$type.']['.$row->id.']">';
				    $html .= '<option value="">'.JText::_('JSELECT').'</option>';
				    $field_options = unserialize($row->field_options);
				    for($i = 0; $i < count($field_options); $i++){
					$selected = ($sel_value==$field_options[$i]) ? ' selected="selected"' : '';
					$html .= '<option value="'.htmlspecialchars($field_options[$i]).'"'.$selected.'>'.$field_options[$i].'</option>';
				    }
				    
				    $html .=    '</select>';
				}
				if($scope=='orderedit'){
					$html .= '</div></div>';
				}
				else {
					$html .=    '</div>';
				}
			}
		}
		
		$result['msg'] = $html;
		echo json_encode($result);
		exit();
	}
	/**
	* Get extras for return section
	*
	* @access	public
	*/
	public function getReturnExtraListAjax()
	{
		$result = array('error' => 0, 'msg' => '');
		$db = JFactory::getDBO();
		$lang =  JFactory::getLanguage();
		$lang_tag = JRequest::getVar('lang', '*');
		
		$id = JRequest::getInt('id', 0);
		
		$booking_type = JRequest::getVar('booking_type', 'address');
		$pickup_type = JRequest::getVar('pickup_type', 'poi');
		$pickup_poi = JRequest::getInt('pickup_poi', 0);
		$pickup_address = JRequest::getVar('pickup_address', '');
		$dropoff_type = JRequest::getVar('dropoff_type', 'poi');
		$dropoff_poi = JRequest::getInt('dropoff_poi', 0);
		$dropoff_address = JRequest::getVar('dropoff_address', '');
		$route_from = JRequest::getInt('route_from', 0);
		$route_to = JRequest::getInt('route_to', 0);
		$scope = JRequest::getVar('scope', 'searchurl');
		
		// get item details if id is present
		$selected_extras = array();
		if($id>0){
			$query = $db->getQuery(true);
			$query->select('*');
			$query->from('#__taxibooking_searchurls');
			$query->where('id = '.(int)$id);
			$db->setQuery((string)$query);
			$urlObj = $db->loadObject();
			
			if($urlObj){
				$params = unserialize($urlObj->url_params);
				$selected_extras = !empty($params['extras']) ? $params['extras'] : array();
			}
		}
		
		$return_pickup_extras = array();
		$return_dropoff_extras = array();
		
		$cwhere = array();
		$cwhere[] = 'f.published = 1';
		$cwhere[] = 'f.show_on_return = 1';
		$cwhere[] = 'f.show_on_user_details = 0';
		
		if($lang_tag!="*"){
			$cwhere[] = '(f.language = "*" OR f.language = "'.$lang_tag.'")';
		}
		
		// in case of return, pickup will be dropoff and dropoff will be pickup
		if($booking_type=='address')
		{
			// get return pickup extras
			$where = array();
			if($dropoff_type=='address'){
				$return_pickup = $dropoff_address;
				$where[] = 'f.show_on_address = 1';
			}
			else {
				$poi_obj = booking_helper::point_details($dropoff_poi);
				$return_pickup = $poi_obj->title;
				$poicat = $poi_obj->catid;
				if($poicat!=""){
					$where[] = "(f.catid='$poicat' OR f.catid LIKE '$poicat,%' OR f.catid LIKE '%,$poicat' OR f.catid LIKE '%,$poicat,%')";
				}
				$where[] = 'f.show_on_pickup = 1';
			}
			
			$where = array_merge($cwhere, $where);
			$where = ( count($where) ? ' WHERE ' . implode(' AND ', $where) : '' );
			$query = 'SELECT f.* '
				. ' FROM #__taxibooking_fields AS f'
				. $where
				. ' ORDER BY f.ordering ASC'
			;
			$db->setQuery($query);
			$db->query();
			$return_pickup_extras = $db->loadObjectList();
			
			// get return dropoff extras
			$where = array();
			if($pickup_type=='address'){
				$return_dropoff = $pickup_address;
				$where[] = 'f.show_on_address = 1';
			}
			else {
				$poi_obj = booking_helper::point_details($pickup_poi);
				$return_dropoff = $poi_obj->title;
				$poicat = $poi_obj->catid;
				if($poicat!=""){
					$where[] = "(f.catid='$poicat' OR f.catid LIKE '$poicat,%' OR f.catid LIKE '%,$poicat' OR f.catid LIKE '%,$poicat,%')";
				}
				$where[] = 'f.show_on_dropoff = 1';
			}
			$where = array_merge($cwhere, $where);
			$where = ( count($where) ? ' WHERE ' . implode(' AND ', $where) : '' );
			$query = 'SELECT f.* '
				. ' FROM #__taxibooking_fields AS f'
				. $where
				. ' ORDER BY f.ordering ASC'
			;
			$db->setQuery($query);
			$db->query();
			$return_dropoff_extras = $db->loadObjectList();
		}
		elseif($booking_type=='offers')
		{
			// get return pickup extras
			$where = array();
			$poi_obj = booking_helper::point_details($route_to);
			$return_pickup = $poi_obj->title;
			$poicat = $poi_obj->catid;
			if($poicat!=""){
				$where[] = "(f.catid='$poicat' OR f.catid LIKE '$poicat,%' OR f.catid LIKE '%,$poicat' OR f.catid LIKE '%,$poicat,%')";
			}
			$where[] = 'f.show_on_pickup = 1';
			
			$where = array_merge($cwhere, $where);
			$where = ( count($where) ? ' WHERE ' . implode(' AND ', $where) : '' );
			$query = 'SELECT f.* '
				. ' FROM #__taxibooking_fields AS f'
				. $where
				. ' ORDER BY f.ordering ASC'
			;
			$db->setQuery($query);
			$db->query();
			$return_pickup_extras = $db->loadObjectList();
			
			// get return dropoff extras
			$where = array();
			$poi_obj = booking_helper::point_details($route_from);
			$return_dropoff = $poi_obj->title;
			$poicat = $poi_obj->catid;
			if($poicat!=""){
				$where[] = "(f.catid='$poicat' OR f.catid LIKE '$poicat,%' OR f.catid LIKE '%,$poicat' OR f.catid LIKE '%,$poicat,%')";
			}
			$where[] = 'f.show_on_dropoff = 1';
			
			$where = array_merge($cwhere, $where);
			$where = ( count($where) ? ' WHERE ' . implode(' AND ', $where) : '' );
			$query = 'SELECT f.* '
				. ' FROM #__taxibooking_fields AS f'
				. $where
				. ' ORDER BY f.ordering ASC'
			;
			$db->setQuery($query);
			$db->query();
			$return_dropoff_extras = $db->loadObjectList();
		}
		
		$html = '';
		if($scope=='orderedit'){
			$html .= '<div class="admin_list_item clearfix address_feature tr_col_from inputWrap">
				<div class="admin_list_item_one">
				    <label for="">'.JText::_( 'SEARCHURL_RETURN_PICKUP' ).':</label>
				</div>
				<div class="admin_list_item_two"><input type="text" id="return_pickup" value="'.$return_pickup.'" class="inputbox" readonly="readonly" /></div>
				</div>';
		}
		else {
			$html .= '<div class="inputWrap address_feature clearfix">
					<label for="title">
					    <span class="">
						'.JText::_( 'SEARCHURL_RETURN_PICKUP' ).':
					    </span>
					</label>
					<input type="text" id="return_pickup" value="'.$return_pickup.'" class="inputbox" readonly="readonly" />
				</div>';
		}
		
		if(!empty($return_pickup_extras))
		{
			foreach($return_pickup_extras as $row)
			{
				$type = 'return_pickup';
				
				if($scope=='orderedit'){
					$html .= '<div class="admin_list_item clearfix tr_col_from inputWrap">
						<div class="admin_list_item_one">
						    <label>'.$row->title.'</label>
						</div>
						<div class="admin_list_item_two">';
				}
				else {
					$html .= '<div class="inputWrap clearfix">
						<label>'.$row->title.'</label>';
				}
				
				$sel_value = '';	
				if(!empty($selected_extras[$type])){
					$sel_value = !empty($selected_extras[$type][$row->id]) ? $selected_extras[$type][$row->id] : '';
				}
			
				if($row->field_type=='input'){
				    $additional_class = '';
				    $html .=    '<input class="inputbox extra'.$additional_class.'" type="text" value="'.$sel_value.'" name="extras['.$type.']['.$row->id.']" />';
				}
				elseif($row->field_type=='textarea'){
					$additional_class = '';
					$html .= '<textarea name="extras['.$type.']['.$row->id.']" class="inputbox extra'.$additional_class.'">'.$sel_value.'</textarea>';
				}
				elseif($row->field_type=='extra'){
				    $html .=    '<select class="inputbox select extra" name="extras['.$type.']['.$row->id.']">';
				    $html .= '<option value="">'.JText::_('JSELECT').'</option>';
				    $start = ($row->allow_zero==1) ? 0 : 1;
				    for($i = $start; $i <= $row->quantity; $i++){
					$selected = ($sel_value==$i) ? ' selected="selected"' : '';
					$html .= '<option value="'.$i.'"'.$selected.'>'.$i.'</option>';
				    }
				    
				    $html .=    '</select>';
				}
				elseif($row->field_type=='optionlist'){
				    $html .=    '<select class="inputbox select extra" name="extras['.$type.']['.$row->id.']">';
				    $html .= '<option value="">'.JText::_('JSELECT').'</option>';
				    $field_options = unserialize($row->field_options);
				    for($i = 0; $i < count($field_options); $i++){
					$selected = ($sel_value==$field_options[$i]) ? ' selected="selected"' : '';
					$html .= '<option value="'.htmlspecialchars($field_options[$i]).'"'.$selected.'>'.$field_options[$i].'</option>';
				    }
				    
				    $html .=    '</select>';
				}
				if($scope=='orderedit'){
					$html .= '</div></div>';
				}
				else {
					$html .=    '</div>';
				}
			}
		}
		
		if($scope=='orderedit'){
			$html .= '<div class="admin_list_item clearfix address_feature tr_col_from inputWrap">
				<div class="admin_list_item_one">
				    <label for="">'.JText::_( 'SEARCHURL_RETURN_DROPOFF' ).':</label>
				</div>
				<div class="admin_list_item_two"><input type="text" id="return_dropoff" value="'.$return_dropoff.'" class="inputbox" readonly="readonly" /></div>
				</div>';
		}
		else {
			$html .= '<div class="inputWrap address_feature clearfix">
					<label for="title">
					    <span class="">
						'.JText::_( 'SEARCHURL_RETURN_DROPOFF' ).':
					    </span>
					</label>
					<input type="text" id="return_dropoff" value="'.$return_dropoff.'" class="inputbox" readonly="readonly" />
				</div>';
		}
		
		if(!empty($return_dropoff_extras))
		{
			foreach($return_dropoff_extras as $row)
			{
				$type = 'return_dropoff';
				
				if($scope=='orderedit'){
					$html .= '<div class="admin_list_item clearfix tr_col_from inputWrap">
						<div class="admin_list_item_one">
						    <label>'.$row->title.'</label>
						</div>
						<div class="admin_list_item_two">';
				}
				else {
					$html .= '<div class="inputWrap clearfix">
						<label>'.$row->title.'</label>';
				}
				
				$sel_value = '';	
				if(!empty($selected_extras[$type])){
					$sel_value = !empty($selected_extras[$type][$row->id]) ? $selected_extras[$type][$row->id] : '';
				}
			
				if($row->field_type=='input'){
				    $additional_class = '';
				    $html .=    '<input class="inputbox extra'.$additional_class.'" type="text" value="'.$sel_value.'" name="extras['.$type.']['.$row->id.']" />';
				}
				elseif($row->field_type=='textarea'){
					$additional_class = '';
					$html .= '<textarea name="extras['.$type.']['.$row->id.']" class="inputbox extra'.$additional_class.'">'.$sel_value.'</textarea>';
				}
				elseif($row->field_type=='extra'){
				    $html .=    '<select class="inputbox select extra" name="extras['.$type.']['.$row->id.']">';
				    $html .= '<option value="">'.JText::_('JSELECT').'</option>';
				    $start = ($row->allow_zero==1) ? 0 : 1;
				    for($i = $start; $i <= $row->quantity; $i++){
					$selected = ($sel_value==$i) ? ' selected="selected"' : '';
					$html .= '<option value="'.$i.'"'.$selected.'>'.$i.'</option>';
				    }
				    
				    $html .=    '</select>';
				}
				elseif($row->field_type=='optionlist'){
				    $html .=    '<select class="inputbox select extra" name="extras['.$type.']['.$row->id.']">';
				    $html .= '<option value="">'.JText::_('JSELECT').'</option>';
				    $field_options = unserialize($row->field_options);
				    for($i = 0; $i < count($field_options); $i++){
					$selected = ($sel_value==$field_options[$i]) ? ' selected="selected"' : '';
					$html .= '<option value="'.htmlspecialchars($field_options[$i]).'"'.$selected.'>'.$field_options[$i].'</option>';
				    }
				    
				    $html .=    '</select>';
				}
				if($scope=='orderedit'){
					$html .= '</div></div>';
				}
				else {
					$html .=    '</div>';
				}
			}
		}
		
		$result['msg'] = $html;
		echo json_encode($result);
		exit();
	}
	/**
        * Method to get dropoff route category based on selected route pickup POI via AJAX.
        *
        * @return  void
        *
        * @since   3.0
        */
	public function getRouteDropoffCategory()
	{
		$result = array('error' => 0, 'msg' => '');
		$db = JFactory::getDBO();
		$session = JFactory::getSession();
		$elsettings = booking_helper::config();
		
		$route_category_pickup = JRequest::getInt('route_pickup_category', 0);
		$route_from = JRequest::getInt('route_pickup_poi', 0);
		$lang_tag = JRequest::getVar('lang', '*');
		
		$route_dropoff_category = $route_dropoff_catarr = array();
		$route_dropoff_category[] = JHTML::_('select.option', 0, JText::_('JSELECT'));
		
		if($route_from > 0)
		{
			$query = 'SELECT DISTINCT poi.title, poi.id, poi_cat.id AS catid, poi_cat.title AS cattitle, poi_cat.image AS cat_image, poi_cat.language AS cat_language'
				. ' FROM #__taxibooking_points AS poi'
				. ' LEFT JOIN #__taxibooking_poicats AS poi_cat ON poi_cat.id = poi.catid'
				. ' LEFT JOIN #__taxibooking_routes AS r ON poi.id = r.route_to'
				. ' WHERE r.published = 1'
				. ' AND poi.published = 1'
				. ' AND poi.show_on_dropoff = 1'
				. ' AND r.route_from = '.$db->Quote($route_from)
				. ' AND (poi.language = "*" OR poi.language = "'.$lang_tag.'")'
				. ' ORDER BY r.ordering ASC'
			;
			$db->setQuery($query);
			$db->query();
			$rows = $db->loadObjectList();
			
			if (!empty($rows))
			{
				foreach ($rows as $row)
				{
					$route_dropoff_catarr[$row->catid] = array('cattitle' => $row->cattitle,
                                                                'catimage' => $row->cat_image);
					
					
				}
			}
		}
		
		if(!empty($route_dropoff_catarr))
		{
			foreach($route_dropoff_catarr as $catid => $catobj)
			{
				$route_dropoff_category[] = JHTML::_('select.option', $catid, $catobj['cattitle']);
			}
		}
		
		$result['msg'] = JHTML::_('select.genericlist', $route_dropoff_category, 'route_dropoff_category', 'class="booking_param"', 'value', 'text', 0 );
		echo json_encode($result);
		exit();
	}
	/**
	 * Get route list
	 *
	 * @access	public
	 */
	public function getRoutePOIList()
	{
		$result = array('error' => 0, 'msg' => '');
		$db = JFactory::getDBO();
		
		$route_category_pickup = JRequest::getInt('route_pickup_category', 0);
		$route_category_dropoff = JRequest::getInt('route_dropoff_category', 0);
		$route_from = JRequest::getInt('route_from', 0);
		$lang_tag = JRequest::getVar('lang', '*');
		
		$route_options = array();
		$route_options[] = JHTML::_('select.option', 0, JText::_('JSELECT'));
		
		if($route_category_dropoff > 0 && $route_from > 0) { // user chose routecategory dropoff, so get the dropoff POI list for this selected category
			
			$field_name = 'route_dropoff_poi';
			
			$query = 'SELECT DISTINCT poi.id, poi.title, poi_cat.image AS cat_image'
				. ' FROM #__taxibooking_points AS poi'
				. ' LEFT JOIN #__taxibooking_poicats AS poi_cat ON poi_cat.id = poi.catid'
				. ' LEFT JOIN #__taxibooking_routes AS r ON poi.id = r.route_to'
				. ' WHERE r.published = 1'
				. ' AND poi.published = 1'
				. ' AND poi.show_on_dropoff = 1'
				. ' AND r.route_from = '.$db->Quote($route_from)
				. ' AND poi_cat.id = '.$db->Quote($route_category_dropoff)
				. ' AND (poi.language = "*" OR poi.language = "'.$lang_tag.'")'
				. ' ORDER BY r.ordering ASC'
			;
		}
		elseif($route_category_pickup > 0 && $route_from == 0) { // user chose pickup routecategory, so get the pickup POI list for this selected category
			
			$field_name = 'route_pickup_poi';
			
			$query = 'SELECT DISTINCT poi.id, poi.title, poi_cat.image AS cat_image'
				. ' FROM #__taxibooking_points AS poi'
				. ' LEFT JOIN #__taxibooking_poicats AS poi_cat ON poi_cat.id = poi.catid'
				. ' LEFT JOIN #__taxibooking_routes AS r ON poi.id = r.route_from'
				. ' WHERE r.published = 1'
				. ' AND poi.published = 1'
				. ' AND poi.show_on_pickup = 1'
				. ' AND poi_cat.id = '.$db->Quote($route_category_pickup)
				. ' AND (poi.language = "*" OR poi.language = "'.$lang_tag.'")'
				. ' ORDER BY r.ordering ASC'
			;
		}
		else {
			$field_name = 'nothing';
		}
		
		if($field_name!="nothing")
		{
			$db->setQuery($query);
			$db->query();
			$rows = $db->loadObjectList();
			
			if (!empty($rows))
			{
				foreach ($rows as $row)
				{
					$row->title = htmlspecialchars($row->title);
					// In case htmlspecialchars misses these.
					$row->title = str_replace(array("'", '"'), array("&#39;", "&quot;"), $row->title);
					$route_options[] = JHTML::_('select.option', $row->id, $row->title);
				}
			}
		}
		
		$result['msg'] =  JHTML::_('select.genericlist', $route_options, $field_name, 'class="booking_param"', 'value', 'text', 0);
		echo json_encode($result);
		exit();
	}
	
	/**
	* Get shuttle dropoff POIs
	*
	* @access	public
	*/
	public function getShuttleDropoffPOIs()
	{
		$result = array('error' => 0, 'msg' => '');
		$db = JFactory::getDBO();
		
		$shuttle_pickup_poi = JRequest::getInt('shuttle_pickup_poi', 0);
		$shuttle_dropoff_poi = JRequest::getInt('shuttle_dropoff_poi', 0);
		$lang_tag = JRequest::getVar('lang', '*');
		
		$shuttle_dropoff_pois = array();
		
		// first collect routes associated with this pickup POI
		$where = array();
		$where[] = 'r.published = 1';
		$where[] = '(r.language = "*" OR r.language = "'.$lang_tag.'")';
		$where[] = 'rs.poi_id = '.$db->Quote($shuttle_pickup_poi);
		
		$where = ( count($where) ? ' WHERE ' . implode(' AND ', $where) : '' );
		$query = 'SELECT rs.*'
			. ' FROM #__taxibooking_shuttle_route_stops AS rs'
			. ' LEFT JOIN #__taxibooking_shuttle_routes AS r ON r.id = rs.route_id'
			. $where
		;
		$db->setQuery($query);
		$db->query();
		$routes = $db->loadObjectList();
		
		if(!empty($routes))
		{
			foreach($routes as $route)
			{
				$pickup_poi_order = $route->ordering;
				
				$where = array();
				$where[] = 'rs.route_id = '.$route->route_id;
				$where[] = 'rs.ordering > '.(int)$pickup_poi_order;
				
				$where = ( count($where) ? ' WHERE ' . implode(' AND ', $where) : '' );
				$query = 'SELECT p.title AS poi_title, p.id AS poiid, p.lat AS poilat, p.long AS poilng, poi_cat.id AS catid, poi_cat.title AS cattitle, poi_cat.image AS cat_image'
					. ' FROM #__taxibooking_shuttle_route_stops AS rs'
					. ' LEFT JOIN #__taxibooking_points AS p ON p.id = rs.poi_id'
					. ' LEFT JOIN #__taxibooking_poicats AS poi_cat ON poi_cat.id = p.catid'
					. $where
					. ' ORDER BY p.title ASC'
				;
				$db->setQuery($query);
				$db->query();
				$rows = $db->loadObjectList();
				
				if(!empty($rows))
				{
					foreach($rows as $row)
					{
						$shuttle_dropoff_pois[$row->poiid] = $row;
					}
				}
			}
		}
		
		$options = array();
		$options[] = JHTML::_('select.option', 0, JText::_('JSELECT'));
		
		if (!empty($shuttle_dropoff_pois))
		{
			foreach ($shuttle_dropoff_pois as $poiid => $row)
			{
				$row->poi_title = htmlspecialchars($row->poi_title);
				// In case htmlspecialchars misses these.
				$row->poi_title = str_replace(array("'", '"'), array("&#39;", "&quot;"), $row->poi_title);
				
				$options[] = JHTML::_('select.option', $row->poiid, $row->poi_title);
			}
		}
		$result['msg'] =  JHTML::_('select.genericlist', $options, 'shuttle_dropoff_poi', 'class="booking_param"', 'value', 'text', $shuttle_dropoff_poi);
		echo json_encode($result);
		exit();
	}
	
	/**
	* Get shuttle dropoff POIs
	*
	* @access	public
	*/
	public function getShuttleTimeOptions()
	{
		$result = array('error' => 0, 'msg' => '');
		$db = JFactory::getDBO();
		$session =  JFactory::getSession();
		$lang =  JFactory::getLanguage();
		$lang_tag = $lang->getTag();
		
		$elsettings =  booking_helper::config();
		
		$siteOffset = booking_helper::set_tbtimezone();
		$dtnow = JFactory::getDate('now', $siteOffset);
		$now = $dtnow->format('YmdHi', true);
		
		$orderdate = JRequest::getVar('shuttle_pickup_date', '');
		$shuttle_pickup_poi = JRequest::getInt('shuttle_pickup_poi', 0);
		$shuttle_dropoff_poi = JRequest::getInt('shuttle_dropoff_poi', 0);
		$shuttle_pickup_time = JRequest::getVar('shuttle_pickup_time', 0);
		$clear_previous_selection = JRequest::getInt('clear_previous_selection', 0);
		
		if($elsettings->date_format=='mm-dd-yy') {
			$date1_arr = explode('-',$orderdate);
			$orderdate =  $date1_arr[2].$date1_arr[0].$date1_arr[1];
		}
		else {
			$date1_arr = explode('-',$orderdate);
			$orderdate =  $date1_arr[2].$date1_arr[1].$date1_arr[0];
		}
		$shuttle_dropoff_pois = array();
		
		$query = 'SELECT DISTINCT t1.route_id, t1.arrival_time'
			. ' FROM (
				SELECT route_id, arrival_time, ordering 
				FROM `#__taxibooking_shuttle_route_stops` rs
				WHERE rs.poi_id = '.(int)$shuttle_pickup_poi
			. ' 	)t1
				JOIN (
				SELECT route_id, arrival_time, ordering 
				FROM `#__taxibooking_shuttle_route_stops` rs
				WHERE rs.poi_id = '.(int)$shuttle_dropoff_poi
			. ' 	)t2'
			. ' WHERE t1.ordering < t2.ordering'
			. ' AND t1.route_id = t2.route_id'
			. ' ORDER BY t1.arrival_time'
		;
		//echo str_replace('#__', 'nfgwj_', $query);
		$db->setQuery($query);
		$db->query();
		$routes = $db->loadObjectList();

		$html = '';
		if(!empty($routes))
		{
			foreach($routes as $route)
			{
				$temp = explode(':', $route->arrival_time);
				//echo $orderdate.$temp[0].$temp[1].'---'.$now.'<br>';
				if($orderdate.$temp[0].$temp[1] > $now)
				{  // hide past times
					if($elsettings->time_format=='12hr'){
						$shuttle_time = date("g:i A", strtotime($route->arrival_time));
					}
					else {
						$shuttle_time = $temp[0].':'.$temp[1];
					}
					
					// get total seats of the shuttle associated with this route
					$query = 'SELECT c.passenger_no AS total_seats'
						. ' FROM #__taxibooking_cars AS c'
						. ' LEFT JOIN #__taxibooking_shuttle_routes AS r ON r.car_id = c.id'
						. ' WHERE r.id = '.(int)$route->route_id
					;
					//echo str_replace('#__', 'nfgwj_', $query);
					$db->setQuery($query);
					$db->query();
					$total_seats = $db->loadResult();
					
					// get total booked seats today of the shuttle associated with this route
					$query = 'SELECT SUM(b.booked_seats) AS booked_seats'
						. ' FROM #__taxibooking_shuttle_bookings AS b'
						. ' WHERE b.route_id = '.(int)$route->route_id
						. ' AND b.order_date = '.$db->Quote($orderdate)
					;
					//echo str_replace('#__', 'nfgwj_', $query);
					$db->setQuery($query);
					$db->query();
					$booked_seats = $db->loadResult();
					
					$available_seats = $total_seats - $booked_seats;
					
					if($clear_previous_selection==1){
						$checked = '';
					}
					else {
						$checked = ($shuttle_pickup_time==$route->arrival_time) ? ' checked="checked"' : '';
					}
					
					$html .= '<div class="shuttletime_wrap clearfix" data-routeid="'.$route->route_id.'" data-seats="'.$available_seats.'">
						<input type="radio" name="shuttletime" class="booking_param" id="shuttletime_'.$route->route_id.'" value="'.$route->arrival_time.'"'.$checked.' />
						<label for="shuttletime_'.$route->route_id.'">'.$shuttle_time.'</label>
						</div>';
				}
			}
		}
		
		$result['msg'] = $html;
		echo json_encode($result);
		exit();
	}
	
	/**
	 * Get route list
	 *
	 * @access	public
	 */
	public function getCouponsAjax()
	{
		$result = array('error' => 0, 'msg' => '');
		$db = JFactory::getDBO();
		
		$booking_type = JRequest::getVar('booking_type', '');
		$id = JRequest::getInt('id', 0);
		$lang_tag = JRequest::getVar('lang', '*');
		
		// get item details if id is present
		$selected_coupon = 0;
		if($id>0){
			$query = $db->getQuery(true);
			$query->select('*');
			$query->from('#__taxibooking_searchurls');
			$query->where('id = '.(int)$id);
			$db->setQuery((string)$query);
			$urlObj = $db->loadObject();
			
			if($urlObj){
				$params = unserialize($urlObj->url_params);
				$selected_coupon = !empty($params['coupons']) ? $params['coupons'] : 0;
			}
		}
		
		$query = $db->getQuery(true);
		$query->select('*');
		$query->from('#__taxibooking_coupons');
		$query->where('published = 1');
		$query->where('for_company_share = 0');
		if($lang_tag!="*"){
			$query->where('(language = "*" OR language = "'.$lang_tag.'")');
		}
		$query->order('ordering ASC');
		$db->setQuery((string)$query);
		$rows = $db->loadObjectList();
		
		$options = array();
		$options[] = JHTML::_('select.option', 0, JText::_('JSELECT'));
		
                if(!empty($rows)){
			foreach ($rows as $row){
				// check booking type
				$allowed_booking_types = ($row->booking_types!="") ? explode(',',$row->booking_types) : array();
				if(!empty($allowed_booking_types) && $booking_type!="" && !in_array($booking_type, $allowed_booking_types)){
					continue;
				}
				$options[] = JHTML::_('select.option',  $row->id, $row->title);
			}
                }
		
		$result['msg'] =  JHTML::_('select.genericlist', $options, 'coupons', 'class="booking_param"', 'value', 'text', (int)$selected_coupon );
		echo json_encode($result);
		exit();
	}
}