<?php
/**
* @title				Minitek FAQ Book
* @copyright   	Copyright (C) 2011-2020 Minitek, All rights reserved.
* @license   		GNU General Public License version 3 or later.
* @author url   https://www.minitek.gr/
* @developers   Minitek.gr
*/

defined('_JEXEC') or die;

use Joomla\Utilities\ArrayHelper;

/**
 * MyQuestion class.
 *
 * @since  4.0.0
 */
class FAQBookProControllerMyQuestion extends JControllerForm
{
	/**
	 * The URL view item variable.
	 *
	 * @var    string
	 * @since  4.0.0
	 */
	protected $view_item = 'myquestion';

	/**
	 * The URL view list variable.
	 *
	 * @var    string
	 * @since  4.0.0
	 */
	protected $view_list = 'myquestions';

	/**
	 * The URL edit variable.
	 *
	 * @var    string
	 * @since  4.0.0
	 */
	protected $urlVar = 'a.id';

	/**
	 * Method to add a new record.
	 *
	 * @return  mixed  True if the record can be added, an error object if not.
	 *
	 * @since   4.0.0
	 */
	public function add()
	{
		if (!parent::add())
		{
			// Redirect to the return page.
			$this->setRedirect($this->getReturnPage());

			return;
		}

		// Redirect to the edit screen.
		$append = '';
		$return = $this->getReturnPage();

		if ($return)
		{
			$append .= '&return=' . base64_encode($return);
		}

		$Itemid = JFactory::getApplication()->input->get('Itemid', '');

		if ($Itemid)
		{
			$append .= '&Itemid=' . $Itemid;
		}

		$this->setRedirect(JRoute::_('index.php?option=com_faqbookpro&view=myquestion&layout=edit&a_id=0'.$append, false));

		return true;
	}

	/**
	 * Method override to check if you can add a new record.
	 *
	 * @param   array  $data  An array of input data.
	 *
	 * @return  boolean
	 *
	 * @since   4.0.0
	 */
	protected function allowAdd($data = array())
	{
		$user       = JFactory::getUser();
		$topicId 	= ArrayHelper::getValue($data, 'topicid', $this->input->getInt('topicid'), 'int');
		$allow      = null;

		if ($topicId)
		{
			// If the topic has been passed in the data or URL check it.
			$allow = $user->authorise('core.create', 'com_faqbookpro.topic.' . $topicId);
		}

		if ($allow === null)
		{
			// In the absense of better information, revert to the component permissions.
			return parent::allowAdd();
		}
		else
		{
			return $allow;
		}
	}

	/**
	 * Method override to check if you can edit an existing record.
	 *
	 * @param   array   $data  An array of input data.
	 * @param   string  $key   The name of the key for the primary key; default is id.
	 *
	 * @return  boolean
	 *
	 * @since   1.6
	 */
	protected function allowEdit($data = array(), $key = 'id')
	{
		$recordId = (int) isset($data[$key]) ? $data[$key] : 0;
		$user = JFactory::getUser();
		$asset    = 'com_faqbookpro.question.' . $recordId;

		// Zero record (id:0), return component edit permission by calling parent controller method
		if (!$recordId)
		{
			return parent::allowEdit($data, $key);
		}

		// Check edit on the record asset (explicit or inherited)
		if ($user->authorise('core.edit', 'com_faqbookpro.question.' . $recordId))
		{
			return true;
		}

		// Check edit own on the record asset (explicit or inherited)
		if ($user->authorise('core.edit.own', 'com_faqbookpro.question.' . $recordId))
		{
			// Existing record already has an owner, get it
			$record = $this->getModel()->getItem($recordId);

			if (empty($record))
			{
				return false;
			}

			// Grant if current user is owner of the record
			return $user->get('id') == $record->created_by;
		}

		return false;
	}

	/**
	 * Method to cancel an edit.
	 *
	 * @param   string  $key  The name of the primary key of the URL variable.
	 *
	 * @return  boolean  True if access level checks pass, false otherwise.
	 *
	 * @since   4.0.0
	 */
	public function cancel($key = 'a_id')
	{
		$result = parent::cancel($key);

		$app = JFactory::getApplication();
		$model = $this->getModel();
		$table = $model->getTable();
		$user = JFactory::getUser();
		$checkin = property_exists($table, 'checked_out');
		$context = "$this->option.edit.$this->context";
		$recordId = $app->input->getInt($key);

		// Redirect to the return page.
		$redirlink = $this->getReturnPage();

		// Attempt to check-in the current record.
		if ($recordId)
		{
			if ($checkin)
			{
				if ($model->checkin($recordId) === false)
				{
					// Check-in failed, go back to the record and display a notice.
					$app->enqueueMessage(JText::sprintf('JLIB_APPLICATION_ERROR_CHECKIN_FAILED', $model->getError()), 'error');

					$this->setRedirect(\JRoute::_($redirlink, false));

					return false;
				}
			}

			// Clean the session data and redirect.
			$this->releaseEditId($context, $recordId);
			$app->setUserState($context . '.data', null);

			$this->setRedirect(\JRoute::_($redirlink, false));

			return false;
		}

		// Clean the session data and redirect.
		$this->releaseEditId($context, $recordId);
		$app->setUserState($context . '.data', null);

		$this->setRedirect(\JRoute::_($redirlink, false));

		return $result;
	}

	/**
	 * Method to edit an existing record.
	 *
	 * @param   string  $key     The name of the primary key of the URL variable.
	 * @param   string  $urlVar  The name of the URL variable if different from the primary key
	 * (sometimes required to avoid router collisions).
	 *
	 * @return  boolean  True if access level check and checkout passes, false otherwise.
	 *
	 * @since   4.0.0
	 */
	public function edit($key = null, $urlVar = 'a_id')
	{
		$app = JFactory::getApplication();
		$model = $this->getModel();
		$table = $model->getTable();
		$user = JFactory::getUser();
		$cid = $this->input->post->get('cid', array(), 'array');

		// Determine the name of the primary key for the data.
		if (empty($key))
		{
			$key = $table->getKeyName();
		}

		// Get the previous record id (if any) and the current record id.
		$recordId = (int) (count($cid) ? $cid[0] : $this->input->getInt($urlVar));

		// Redirect to the return page.
		$redirlink = $this->getReturnPage();

		// Access check.
		if (!$user->id || !$this->allowEdit(array($key => $recordId), $key))
		{
			$app->enqueueMessage(JText::_('JLIB_APPLICATION_ERROR_EDIT_NOT_PERMITTED'), 'error');

			$this->setRedirect(\JRoute::_($redirlink, false));

			return false;
		}
		else
		{
			// Redirect to the edit screen.
			$append = '';
			$return = $redirlink;

			if ($return)
			{
				$append .= '&return=' . base64_encode($return);
			}

			$Itemid = JFactory::getApplication()->input->get('Itemid', '');

			if ($Itemid)
			{
				$append .= '&Itemid=' . $Itemid;
			}

			$this->setRedirect(JRoute::_('index.php?option=com_faqbookpro&view=myquestion&layout=edit&a_id='.$recordId.''.$append, false));

			return true;
		}
	}

	/**
	 * Method to get a model object, loading it if required.
	 *
	 * @param   string  $name    The model name. Optional.
	 * @param   string  $prefix  The class prefix. Optional.
	 * @param   array   $config  Configuration array for model. Optional.
	 *
	 * @return  object  The model.
	 *
	 * @since   4.0.0
	 */
	public function getModel($name = 'Myquestion', $prefix = '', $config = array('ignore_request' => true))
	{
		return parent::getModel($name, $prefix, $config);
	}

	/**
	 * Method to save a record.
	 *
	 * @param   string  $key     The name of the primary key of the URL variable.
	 * @param   string  $urlVar  The name of the URL variable if different from the primary key (sometimes required to avoid router collisions).
	 *
	 * @return  boolean  True if successful, false otherwise.
	 *
	 * @since   4.0.0
	 */
	public function save($key = null, $urlVar = 'a_id')
	{
		// Check for request forgeries.
		JSession::checkToken() or jexit(JText::_('JINVALID_TOKEN'));

		$app = JFactory::getApplication();
		$lang = JFactory::getLanguage();
		$model = $this->getModel();
		$table = $model->getTable();
		$user = JFactory::getUser();
		$data = $this->input->post->get('jform', array(), 'array');
		$checkin = property_exists($table, 'checked_out');
		$context = "$this->option.edit.$this->context";
		$task = $this->getTask();
		$params = JComponentHelper::getParams('com_faqbookpro');

		// Determine the name of the primary key for the data.
		if (empty($key))
		{
			$key = $table->getKeyName();
		}

		// To avoid data collisions the urlVar may be different from the primary key.
		if (empty($urlVar))
		{
			$urlVar = $key;
		}

		$recordId = $this->input->getInt($urlVar);

		// Populate the row id from the session.
		$data[$key] = $recordId;

		// Redirect to the return page.
		$redirlink = $this->getReturnPage();

		// Redirect
		$append = '';
		$return = $redirlink;

		if ($return)
		{
			$append .= '&return=' . base64_encode($return);
		}

		$Itemid = JFactory::getApplication()->input->get('Itemid', '');

		if ($Itemid)
		{
			$append .= '&Itemid=' . $Itemid;
		}

		// Access check.
		if (!$this->allowSave($data, $key))
		{
			$app->enqueueMessage(JText::_('JLIB_APPLICATION_ERROR_SAVE_NOT_PERMITTED'), 'error');

			if (!$recordId)
			{
				$this->setRedirect(\JRoute::_($redirlink, false));
			}
			else
			{
				$this->setRedirect(JRoute::_(FaqBookProHelperRoute::getQuestionRoute($recordId), false));

				return false;
			}
		}

		// Validate the posted data.
		// Sometimes the form needs some posted data, such as for plugins and modules.
		$form = $model->getForm($data, false);

		if (!$form)
		{
			$app->enqueueMessage($model->getError(), 'error');

			return false;
		}

		// Test whether the data is valid.
		$validData = $model->validate($form, $data);

		// Check for validation errors.
		if ($validData === false)
		{
			// Get the validation messages.
			$errors = $model->getErrors();

			// Push up to three validation messages out to the user.
			for ($i = 0, $n = count($errors); $i < $n && $i < 3; $i++)
			{
				if ($errors[$i] instanceof Exception)
				{
					$app->enqueueMessage($errors[$i]->getMessage(), 'warning');
				}
				else
				{
					$app->enqueueMessage($errors[$i], 'warning');
				}
			}

			// Save the data in the session.
			$app->setUserState($context . '.data', $data);

			// Redirect back to the edit screen.
			$this->setRedirect(JRoute::_('index.php?option=com_faqbookpro&view=myquestion&layout=edit&a_id='.$recordId.''.$append, false));

			return false;
		}

		// Attempt to save the data.
		$row_id = $model->save($validData);
		
		if (!$row_id)
		{
			// Save the data in the session.
			$app->setUserState($context . '.data', $validData);

			// Redirect back to the edit screen.
			$app->enqueueMessage(JText::sprintf('JLIB_APPLICATION_ERROR_SAVE_FAILED', $model->getError()), 'error');

			$this->setRedirect(JRoute::_('index.php?option=com_faqbookpro&view=myquestion&layout=edit&a_id='.$recordId.''.$append, false));

			return false;
		}

		// Save succeeded, so check-in the record.
		if ($checkin && $model->checkin($validData[$key]) === false)
		{
			// Save the data in the session.
			$app->setUserState($context . '.data', $validData);

			// Check-in failed, so go back to the record and display a notice.
			$app->enqueueMessage(JText::sprintf('JLIB_APPLICATION_ERROR_CHECKIN_FAILED', $model->getError()), 'error');

			// Get recordId
			if (!$recordId)
			{
				 $recordId = $row_id;
			}

			if (is_int($recordId))
			{
				$this->setRedirect(JRoute::_(FaqBookProHelperRoute::getQuestionRoute($recordId), false));
			}
			else
			{
				$this->setRedirect(JRoute::_(FaqBookProHelperRoute::getTopicRoute($validData['topicid']), false));
			}

			return false;
		}

		if ($params->get('auto_publish', 0))
		{
			$this->setMessage(
				JText::_('COM_FAQBOOKPRO_QUESTION_SUCCESSFULLY_SUBMITED')
			);
		}
		else
		{
			$this->setMessage(
				JText::_('COM_FAQBOOKPRO_QUESTION_SUCCESSFULLY_SUBMITED_REVIEW_PENDING')
			);
		}

		// Redirect the user and adjust session state based on the chosen task.
		switch ($task)
		{
			default:
				// Clear the record id and data from the session.
				$this->releaseEditId($context, $recordId);
				$app->setUserState($context . '.data', null);

				// Get recordId
				if (!$recordId)
				{
					 $recordId = $row_id;
				}

				if (is_int($recordId))
				{
					$this->setRedirect(JRoute::_(FaqBookProHelperRoute::getQuestionRoute($recordId), false));
				}
				else
				{
					$this->setRedirect(JRoute::_(FaqBookProHelperRoute::getTopicRoute($validData['topicid']), false));
				}

				break;
		}

		// Invoke the postSave method to allow for the child class to access the model.
		$this->postSaveHook($model, $validData);

		return true;
	}

	/**
	 * Get the return URL.
	 *
	 * If a "return" variable has been passed in the request
	 *
	 * @return  string	The return URL.
	 *
	 * @since   4.0.0
	 */
	protected function getReturnPage()
	{
		if (JFactory::getApplication()->getUserState( 'com_faqbookpro.return_page'))
		{
			return base64_decode(JFactory::getApplication()->getUserState( 'com_faqbookpro.return_page'));
		}
		else
		{
			if ($this->input->get('return', null, 'base64'))
			{
				return base64_decode($this->input->get('return', null, 'base64'));
			}
			else
			{
				return \JUri::base();
			}
		}
	}
}
