<?php
/** 
	* @package		Custom Canonical
	* @version		1.2.1
	* @copyright	Copyright (C) 2018 DanielMorell.com. All rights reserved. 
	* @license		http://www.gnu.org/licenses/gpl-3.0.html GNU/GPLv3 or later
*/

use Joomla\CMS\Router\Route;

class PlgCustomCanonicalHelper
{

    public $schemes = array("//" , "http://" , "https://");

    public $types = array(
        'article' => array(
            'form_selector' => '#jform_attribs_canonical_url',
            'url_base' => 'index.php?option=com_content&view=article'
        ),
        'category' => array(
            'form_selector' => '#jform_params_canonical_url',
            'url_base' => 'index.php?option=com_content&view=category'
        ),
        'tag' => array(
            'form_selector' => '#jform_params_canonical_url',
            'url_base' => 'index.php?option=com_tags&view=tag'
        ),
        'menu' => array(
            'form_selector' => '#jform_params_canonical_url',
            'url_base' => 'index.php?'
        )
    );

    public $js_selectors = array('$field$', '$disabled$', '$url$', '$DISABLED$', '$ENABLED$');

    /**
     * Adds the plugin root URL to relative canonical URLs.
     *
     * @param   string  $url A string containing the canonical URL.
     * @param   object  $params An object containing the plugin parameters.
     * @return  string  $url A string containing the final absolute canonical URL.
     * @access  public
     */
	
	public function buildUrl($url, $params) {
		if (!empty($params->get('root_url', ''))) {
            $root_url = $params->get('root_url', '');
            if ($this->isAbsoluteUrl($url, $this->schemes)) {
                return $url;
            } else {
                if (substr($url, 0, 1) !== '/') {
                    $url = '/' . $url;
                }
                if (substr($root_url, -1) === '/') {
                    $root_url = rtrim($root_url, '/');
                }
                $url = $root_url . $url;
            }
        }
		return $url;
    }

    /**
     * Adds the plugin root URL to relative canonical URLs.
     *
     * @param   string   $url A string containing the canonical URL.
     * @param   array    $schemes An array containing the possible schemes.
     * @return  boolean  True if absolute URL, false otherwise.
     */

    function isAbsoluteUrl($url, array $schemes) {
        foreach($schemes as $scheme) {
            if (substr($url, 0, strlen($scheme)) === $scheme) {
                return true;
            }
        }
        return false;
    }

    /**
     * This function compiles and registers the auto_canonical JS.
     * 
     * @param  string $type The type of content that the canonical should be created for.
     * @param  object $data The form data object.
     * @return null
     */
    public function autoLoadCanonicalBtn($type, $data) {

        $doc       = JFactory::getDocument();
        $type_info = $this->types[$type];
        $url       = $this->buildAutoURL($type, $data, $type_info['url_base']);
        $js_values = array(
            $type_info['form_selector'],
            ($data->id != null)? 'false': 'true',
            $url,
            JText::_('CUSTOM_CANONICAL_AUTO_DISABLED'),
            JText::_('CUSTOM_CANONICAL_AUTO_ENABLED')
        );
        $auto_canonical_js = file_get_contents(dirname(__FILE__) . '/import/js/auto_canonical.js');
        $auto_canonical_js = str_replace($this->js_selectors, $js_values, $auto_canonical_js);
        $doc->addScriptDeclaration($auto_canonical_js);
    }

    /**
     * This method creates the SEF URL for the auto_canonical JS.
     * 
     * @param  string $type     The type of content that the canonical should be created for.
     * @param  object $data     The form data object.
     * @param  string $base_url The base URL for the specified `$type`.
     * @return string $url      The SEF URL for the current form data.
     */
    function buildAutoURL($type, $data, $base_url) {
        $url = $base_url;
        switch ($type) {
            case 'article':
                $url_params = array(
                    'id' => $data->id,
                    'catid' => $data->catid
                );
                break;
            
            case 'category':
                $url_params = array(
                    'id' => $data->id
                );
                break;
        
            case 'tag':
                $url_params = array(
                    'id' => $data->id . ':' . $data->alias
                );
                $item_id = $this->getTagItemID($data->id);
                if ($item_id > 0) {
                    $url_params['Itemid'] = $item_id;
                }
                break;
        
            case 'menu':
                $url_params = array(
                    'Itemid' => $data->id
                );
                break;
        }
        $n = 0;
        // Add URL params to base URL
        foreach ($url_params as $name => $value) {
            if (substr_compare($url, '?', -1, 1) === 0 && $n == 0) {
                $url .= $name . '=' . $value;
            } else {
                $url .= '&' . $name . '=' . $value;
            }
        }
        // Make URL SEF and return
        return Route::link('site', $url);
    }

    /**
     * This method queries the DB for the tag menu Itemid.
     * 
     * @param  int $id      The ID of the current tag.
     * @return int $menu_id The menu Itemid for the current tag.
     */
    function getTagItemID($id) {
        // Query DB for com_tag menu items
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);
        $query->select($db->quoteName(array('id', 'link')));
        $query->from($db->quoteName('#__menu'));
        $query->where(
            $db->quoteName('link') . ' LIKE ' . $db->quote('index.php?option=com_tags&view=tag%') . ' AND ' . 
            $db->quoteName('published') . ' = 1 AND ' .
            $db->quoteName('client_id') . ' = 0'
        );
        $db->setQuery($query);
        $results = $db->loadObjectList();

        // Filter results for best menu item
        $found = 0;
        $menu_id = 0;
        foreach( $results as $menu_item ) {
            parse_str(parse_url($menu_item->link)['query'], $params);
            if ($found === 0 && $menu_item->link == 'index.php?option=com_tags&view=tags') {
                $menu_id = $menu_item->id;
                $found = 1;
            } elseif ($found < 2 && isset($params['view']) && $params['view'] == 'tag' && isset($params['id']) && in_array($id, $params['id'])) {
                $menu_id = $menu_item->id;
                $found = 2;
            }
        }
        return $menu_id;
    }
}