<?php
/**
 * @package     Joomla.Administrator
 * @subpackage  com_taxibooking
 *
 * @copyright   Copyright (C) 2005 - 2012 Open Source Matters, Inc. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined( '_JEXEC' ) or die( 'Restricted access' );

if(!class_exists('tbPaymentPlugin')) {
	require(JPATH_ROOT.DS.'components'.DS.'com_taxibooking'.DS.'classes'.DS.'tbpayment_plugin_helper.php');
}

class plgTbPaymentPaypal extends tbPaymentPlugin {

	var $_name = 'paypal';
	
	// instance of class
	public static $_this = FALSE;

	function __construct (& $subject, $config) {
		
		parent::__construct ($subject, $config);

		$this->_loggable = TRUE;
		$this->tableFields = array_keys ($this->getTableSQLFields ());
		$this->_tablepkey = 'id'; ;
		$this->_tableId = 'id'; ;
		$varsToPush = array('paypal_merchant_email'  => array('', 'char'),
		                    'paypal_verified_only'   => array('', 'int'),
		                    'payment_currency'       => array('', 'int'),
		                    'sandbox'                => array(0, 'int'),
		                    'sandbox_merchant_email' => array('', 'char'),
		                    'payment_logos'          => array('', 'char'),
		                    'debug'                  => array(0, 'int'),
		                    'status_pending'         => array('', 'char'),
		                    'status_success'         => array('', 'char'),
		                    'status_canceled'        => array('', 'char'),
				    'convert_currency'       => array(0, 'int'),
				    'convert_from'           => array('', 'char'),
				    'convert_to'             => array('', 'char'),
				    'conversion_index'       => array(0, 'float'),
		                    'countries'              => array('', 'char'),
		                    'min_amount'             => array('', 'int'),
		                    'max_amount'             => array('', 'int'),
		                    'secure_post'            => array('', 'int'),
		                    'ipn_test'               => array('', 'int'),
		                    'no_shipping'            => array('', 'int'),
		                    'address_override'       => array('', 'int'),
		                    'cost_per_transaction'   => array('', 'int'),
		                    'cost_percent_total'     => array('', 'int'),
				    'prepayment_percent'     => array('', 'int'),
		                    'tax_id'                 => array(0, 'int')
		);
		
		$this->currencies = array(
			'AUD' => array(
				'label' => 'Australian Dollar',
				'format' => ' ($%s)',
			),
			'CAD' => array(
				'label' => 'Canadian Dollar',
				'format' => ' ($%s)',
			),
			'EUR' => array(
				'label' => 'Euro',
				'format' => ' (€%s)',
			),
			'GBP' => array(
				'label' => 'Pound Sterling',
				'format' => ' (£%s)',
			),
			'JPY' => array(
				'label' => 'Japanese Yen',
				'format' => ' (¥%s)',
			),
			'USD' => array(
				'label' => 'U.S. Dollar',
				'format' => ' ($%s)',
			),
			'NZD' => array(
				'label' => 'N.Z. Dollar',
				'format' => ' ($%s)',
			),
			'CHF' => array(
				'label' => 'Swiss Franc',
				'format' => ' (%s Fr)',
			),
			'HKD' => array(
				'label' => 'Hong Kong Dollar',
				'format' => ' ($%s)',
			),
			'SGD' => array(
				'label' => 'Singapore Dollar',
				'format' => ' ($%s)',
			),
			'SEK' => array(
				'label' => 'Swedish Krona',
				'format' => ' (%s kr)',
			),
			'DKK' => array(
				'label' => 'Danish Krone',
				'format' => ' (%s kr)',
			),
			'PLN' => array(
				'label' => 'Polish Zloty',
				'format' => ' (%s zł)',
			),
			'NOK' => array(
				'label' => 'Norwegian Krone',
				'format' => ' (%s kr)',
			),
			'HUF' => array(
				'label' => 'Hungarian Forint',
				'format' => ' (%s Ft)',
			),
			'CZK' => array(
				'label' => 'Czech Koruna',
				'format' => ' (%s Kč)',
			),
			'ILS' => array(
				'label' => 'Israeli New Sheqel',
				'format' => ' (₪ %s)',
			),
			'MXN' => array(
				'label' => 'Mexican Peso',
				'format' => ' ($%s)',
			),
			'BRL' => array(
				'label' => 'Brazilian Real',
				'format' => ' (R$ %s)',
			),
			'MYR' => array(
				'label' => 'Malaysian Ringgit',
				'format' => ' (RM %s)',
			),
			'PHP' => array(
				'label' => 'Philippine Peso',
				'format' => ' (₱ %s)',
			),
			'RUB' => array(
				'label' => 'Russian Ruble',
				'format' => ' (%s руб)',
			),
			'TWD' => array(
				'label' => 'New Taiwan Dollar',
				'format' => ' (NT$ %s)',
			),
			'THB' => array(
				'label' => 'Thai Baht',
				'format' => ' (฿ %s)',
			),
			'TRY' => array(
				'label' => 'Turkish Lira',
				'format' => ' (TRY %s)', // Unicode is ₺ but this doesn't seem to be widely supported yet (introduced Sep 2012)
			),
		);
	}

	/**
	 * @return string
	 */
	public function getTbPluginCreateTableSQL () {

		return $this->createTableSQL ('Payment Paypal Table');
	}

	/**
	 * @return array
	 */
	function getTableSQLFields () {

		$SQLfields = array(
			'id'                                     => 'int(11) UNSIGNED NOT NULL AUTO_INCREMENT',
			'order_id'                    		=> 'int(1) UNSIGNED',
			'order_number'                           => 'char(64)',
			'paymentmethod_id'            		=> 'mediumint(1) UNSIGNED',
			'payment_name'                           => 'varchar(5000)',
			'payment_order_total'                    => 'decimal(15,5) NOT NULL DEFAULT \'0.00000\'',
			'payment_currency'                       => 'char(3) ',
			'cost_per_transaction'                   => 'decimal(10,2)',
			'cost_percent_total'                     => 'decimal(10,2)',
			'prepayment_percent'                     => 'decimal(10,2)',
			'paypal_custom'                          => 'varchar(255)',
			'paypal_response_mc_gross'               => 'decimal(10,2) NULL DEFAULT NULL',
			'paypal_response_mc_currency'            => 'char(10) NULL DEFAULT NULL',
			'paypal_response_invoice'                => 'char(32) NULL DEFAULT NULL',
			'paypal_response_protection_eligibility' => 'char(128) NULL DEFAULT NULL',
			'paypal_response_payer_id'               => 'char(13) NULL DEFAULT NULL',
			'paypal_response_tax'                    => 'decimal(10,2) NULL DEFAULT NULL',
			'paypal_response_payment_date'           => 'char(28) NULL DEFAULT NULL',
			'paypal_response_payment_status'         => 'char(50) NULL DEFAULT NULL',
			'paypal_response_pending_reason'         => 'char(50) NULL DEFAULT NULL',
			'paypal_response_mc_fee'                 => 'decimal(10,2) NULL DEFAULT NULL',
			'paypal_response_payer_email'            => 'char(128) NULL DEFAULT NULL',
			'paypal_response_last_name'              => 'char(64) NULL DEFAULT NULL',
			'paypal_response_first_name'             => 'char(64) NULL DEFAULT NULL',
			'paypal_response_business'               => 'char(128) NULL DEFAULT NULL',
			'paypal_response_receiver_email'         => 'char(128) NULL DEFAULT NULL',
			'paypal_response_transaction_subject'    => 'char(128) NULL DEFAULT NULL',
			'paypal_response_residence_country'      => 'char(2) NULL DEFAULT NULL',
			'paypal_response_txn_id'                 => 'char(32) NULL DEFAULT NULL',
			'paypal_response_txn_type'               => 'char(32) NULL DEFAULT NULL', //The kind of transaction for which the IPN message was sent
			'paypal_response_parent_txn_id'          => 'char(32) NULL DEFAULT NULL',
			'paypal_response_case_creation_date'     => 'char(32) NULL DEFAULT NULL',
			'paypal_response_case_id'                => 'char(32) NULL DEFAULT NULL',
			'paypal_response_case_type'              => 'char(32) NULL DEFAULT NULL',
			'paypal_response_reason_code'            => 'char(32) NULL DEFAULT NULL',
			'paypalresponse_raw'                     => 'varchar(512) NULL DEFAULT NULL',
		);
		return $SQLfields;
	}
	
	/**
	 * plgTbDisplayListFEPayment
	 * This event is fired to display the pluginmethods in the cart (edit shipment/payment) for exampel
	 *
	 * @param integer $selected ID of the method selected
	 * @return boolean True on succes, false on failures, null when this plugin was not selected.
	 * On errors, JError::raiseWarning (or JError::raiseError) must be used to set a message.
	 */
	public function plgTbDisplayListFEPayment ($selected = 0, &$htmlIn) {
		return $this->displayListFE ($selected, $htmlIn);
	}
	
	/**
	 * @param                $method
	 * @param                $cart_prices
	 * @return int
	 */
	function getCosts ($method, $cart_price) {

		if (preg_match ('/%$/', $method->cost_percent_total)) {
			$cost_percent_total = substr ($method->cost_percent_total, 0, -1);
		} else {
			$cost_percent_total = $method->cost_percent_total;
		}
		return ($method->cost_per_transaction + ($cart_price * $cost_percent_total * 0.01));
	}
	
	/**
	 * Check if the payment conditions are fulfilled for this payment method
	 *
	 */
	protected function checkConditions ($method, $cart_price) {

		$this->convert ($method);

		$amount = $cart_price;
		
		$amount_cond = ($amount >= $method->min_amount AND $amount <= $method->max_amount
			OR
			($method->min_amount <= $amount AND ($method->max_amount == 0)));
		
		if (!$amount_cond) {
			return FALSE;
		}
		
		return TRUE;
	}

	function convert ($method) {

		$method->min_amount = (float)$method->min_amount;
		$method->max_amount = (float)$method->max_amount;
	}
	
	/**
	 * @param array          $cart_prices
	 * @param                $cart_prices_name
	 * @return bool|null
	 */
	public function plgTbonSelectedCalculatePricePayment ($method_id, $cart_prices, &$cart_prices_name) {

		return $this->onSelectedCalculatePrice($method_id, $cart_prices, $cart_prices_name);
	}
	
	/**
	 * @param array          $cart_prices
	 * @param                $cart_prices_name
	 * @return bool|null
	 */
	public function plgTbonSelectedShowPriceLabel ($method_id, $cart_prices, &$payment_labels) {
		
		if (!($method = $this->selectedThisByMethodId ($method_id))) {
			return NULL; // Another method was selected, do nothing
		}
		
		$prepay = $this->_getPrePaymentPercent( $method );
		
		if($prepay<100)
		{
			$cost = $this->getCosts($method, $cart_prices);
			$total = $cart_prices + $cost;
			$due_now = $total * $prepay / 100;
			$due_later = $total - $due_now;
			
			$converted_currency_lbl = '';
			if((int)$method->convert_currency==1 && $method->convert_to!=""){
				$converted = $due_now * (float)$method->conversion_index;
				$payment_currency = $method->convert_to;
				
				// Hungarian Forint = HUF, Japanese Yen = JPY, Taiwan New Dolla = TWD 
				// Note Decimal amounts are not supported for these currency. Passing a decimal amount will throw an error.
				$special = array('HUF', 'JPY', 'TWD');
				if(in_array($method->convert_to, $special)){
					$converted = round($converted);
				}
				else {
					$converted = number_format($converted, 2);
				}
				$converted_currency_lbl .= JText::sprintf($this->currencies[$payment_currency]['format'], $converted);
			}
			
			$payment_labels .= '<div class="check_box_wrap clearfix last">
						<div class="check_name">'.JText::_('TBPAYMENT_PAYPAL_PREPAYMENT_DUE_NOW').':</div>
						<div class="check_desc">'.booking_helper::price_display($due_now).$converted_currency_lbl.'</div>
					</div>';
			$payment_labels .= '<div class="check_box_wrap clearfix last">
						<div class="check_name">'.JText::_('TBPAYMENT_PAYPAL_PREPAYMENT_DUE_LATER').':</div>
						<div class="check_desc">'.booking_helper::price_display($due_later).'</div>
					</div>';
		}
		
		return TRUE;
	}
	
	/**
	 * @param array          $cart_prices
	 * @param                $cart_prices_name
	 * @return bool|null
	 */
	public function plgTbonShowConvertedPriceLabel ($method_id, $grand_total, &$converted_currency_label) {
		
		if (!($method = $this->selectedThisByMethodId ($method_id))) {
			return NULL; // Another method was selected, do nothing
		}
		
		$prepay = $this->_getPrePaymentPercent( $method );
		
		if($prepay==100)
		{
			if((int)$method->convert_currency==1 && $method->convert_to!=""){
				$converted = $grand_total * (float)$method->conversion_index;
				$payment_currency = $method->convert_to;
				
				// Hungarian Forint = HUF, Japanese Yen = JPY, Taiwan New Dolla = TWD 
				// Note Decimal amounts are not supported for these currency. Passing a decimal amount will throw an error.
				$special = array('HUF', 'JPY', 'TWD');
				if(in_array($method->convert_to, $special)){
					$converted = round($converted);
				}
				else {
					$converted = number_format($converted, 2);
				}
				$converted_currency_label .= JText::sprintf($this->currencies[$payment_currency]['format'], $converted);
			}
		}
		
		return TRUE;
	}
	
	/**
	 * @param $plugin plugin
	 */
	protected function renderPluginName ($plugin) {

		$return = '';
		$plugin_name = 'title';
		$plugin_desc = 'text';
		$description = '';
		
		if (!empty($plugin->$plugin_desc)) {
			$description = '<span class="payment_description">' . $plugin->$plugin_desc . '</span>';
		}
		
		$prepayment_text = '';
		$prepay = $this->_getPrePaymentPercent( $plugin );
		if($prepay<100){
			$prepayment_text = ' ('.JText::sprintf('TBPAYMENT_PAYPAL_PREPAYMENT_PERCENT_TEXT', $prepay). '%)';
		}
		
		$pluginName = $return . '<span class="payment_name">' . $plugin->$plugin_name .$prepayment_text. '</span>' . $description;
		return $pluginName;
	}
	
	protected function getGrandTotalLabel ($method, $selectedPlugin, $pluginSalesPrice) {
		$html = '';
		$session =  JFactory::getSession();
		$pluginmethod_id = $this->_idName;
		$sub_total = $session->get('price', 0);
		
		if (preg_match ('/%$/', $method->cost_percent_total)) {
			$cost_percent_total = substr ($method->cost_percent_total, 0, -1);
		} else {
			$cost_percent_total = $method->cost_percent_total;
		}
		$flat_fee = $method->cost_per_transaction;
		$percentage_fee = $sub_total * $cost_percent_total * 0.01;
		
		$grand_total = $sub_total + $flat_fee + $percentage_fee;

		$html .= JText::sprintf('USER_DETAILS_PAYMENTS_GRAND_TOTAL_TXT', booking_helper::price_display($grand_total));
		
		if((int)$method->convert_currency==1 && $method->convert_to!=""){
			$converted = $grand_total * (float)$method->conversion_index;
			$payment_currency = $method->convert_to;
			
			// Hungarian Forint = HUF, Japanese Yen = JPY, Taiwan New Dolla = TWD 
			// Note Decimal amounts are not supported for these currency. Passing a decimal amount will throw an error.
			$special = array('HUF', 'JPY', 'TWD');
			if(in_array($method->convert_to, $special)){
				$converted = round($converted);
			}
			else {
				$converted = number_format($converted, 2, '.', '');
			}
			
			$html .= JText::sprintf($this->currencies[$payment_currency]['format'], $converted);
		}
		
		$prepay = $this->_getPrePaymentPercent( $method );
		if($prepay<100){
			$due_now = $grand_total * $prepay / 100;
			$due_later = $grand_total - $due_now;
			$html .= ' ('.JText::sprintf('TBPAYMENT_PAYPAL_PREPAYMENT_PERCENT_TEXT', $prepay). '% - '.booking_helper::price_display($due_now).')';
		}

		return $html;
	}
	
	/**
	 * @param $cart
	 * @param $order
	 * @return bool|null
	 */
	public function plgTbOrderSubmit ($order) {

		if (!($method = $this->getPluginMethod ($order->payment))) {
			return NULL; // Another method was selected, do nothing
		}
		if (!$this->selectedThisElement ($method->payment_element)) {
			return FALSE;
		}
		
		$app = JFactory::getApplication();
		$session = JFactory::getSession ();
		$return_context = $session->getId ();
		// Get the page/component configuration
		$elsettings = booking_helper::config();
		
		$merchant_email = $this->_getMerchantEmail ($method);
		if (empty($merchant_email)) {
			$app->enqueueMessage (JText::_ ('TBPAYMENT_PAYPAL_MERCHANT_EMAIL_NOT_SET'));
			return FALSE;
		}
		
		$prepay = $this->_getPrePaymentPercent( $method );
		$order_total = $order->cprice;
		$amount = $order_total * $prepay / 100;
		
		$payment_currency = $elsettings->currency;
		$cost_per_transaction = $method->cost_per_transaction;
		$cost_percent_total = $method->cost_percent_total;
		// convert currency
		if((int)$method->convert_currency==1 && $method->convert_to!=""){
			$order_total = $order_total * (float)$method->conversion_index;
			$amount = $amount * (float)$method->conversion_index;
			$payment_currency = $method->convert_to;
		}

		// Prepare data that should be stored in the database		
		$dbValues['payment_name'] = $this->renderPluginName( $method);
		$dbValues['order_id'] = $order->id;
		$dbValues['order_number'] = $order->order_number;
		$dbValues['paymentmethod_id'] = $order->payment;
		$dbValues['paypal_custom'] = $return_context;
		$dbValues['cost_per_transaction'] = $cost_per_transaction;
		$dbValues['cost_percent_total'] = $cost_percent_total;
		$dbValues['prepayment_percent'] = $prepay;
		$dbValues['payment_currency'] = $payment_currency;
		$dbValues['payment_order_total'] = $order_total;
		
		$this->storePluginInternalData ($dbValues);
		
		return TRUE;
	}
	
	/**
	 * This event triggers after order submit, redrects user to payment gateway
	 */
	public function plgTbProcessPayment($order){
		
		if (!($method = $this->getPluginMethod ($order->payment))) {
			return NULL; // Another method was selected, do nothing
		}
		if (!$this->selectedThisElement ($method->payment_element)) {
			return FALSE;
		}
		
		$app = JFactory::getApplication();
		$session = JFactory::getSession ();
		$return_context = $session->getId ();
		// Get the page/component configuration
		$elsettings = booking_helper::config();
		
		$merchant_email = $this->_getMerchantEmail ($method);
		if (empty($merchant_email)) {
			$app->enqueueMessage (JText::_ ('TBPAYMENT_PAYPAL_MERCHANT_EMAIL_NOT_SET'));
			return FALSE;
		}
		
		$begin = $order->begin;
		$end = $order->end;
		$order_date = booking_helper::get_order_pickup_date($order, $elsettings);
		
		if($order->booking_type=='hourly'){
			$itemname = JText::sprintf('TBPAYMENT_PAYPAL_HOURLY_ITEM_NAME', $order_date);
		}
		else {
			$itemname = JText::sprintf('TBPAYMENT_PAYPAL_ADDRESS_ITEM_NAME', $begin, $end, $order_date);
		}
		
		if ($order->returntrip==1)
		{
			$order_return_date = booking_helper::get_order_return_date($order, $elsettings);
			$itemname .= JText::sprintf('TBPAYMENT_PAYPAL_ITEM_NAME_RETURN', $order_return_date);
		}
		
		$prepay = $this->_getPrePaymentPercent( $method );
		$amount = $order->cprice * $prepay / 100;
		
		$payment_currency = $elsettings->currency;
		// convert currency
		if((int)$method->convert_currency==1 && $method->convert_to!=""){
			$amount = $amount * (float)$method->conversion_index;
			$payment_currency = $method->convert_to;
			
			// Hungarian Forint = HUF, Japanese Yen = JPY, Taiwan New Dolla = TWD 
			// Note Decimal amounts are not supported for these currency. Passing a decimal amount will throw an error.
			$special = array('HUF', 'JPY', 'TWD');
			if(in_array($method->convert_to, $special)){
				$amount = round($amount);
			}
			else {
				$amount = number_format($amount, 2, '.', '');
			}
		}
		else {
			$amount = number_format($amount, 2, '.', '');
		}

		$post_variables = Array(
			'cmd'              => '_ext-enter',
			'redirect_cmd'     => '_xclick',
			'upload'           => '1', //Indicates the use of third-party shopping cart
			'charset'	   => 'utf-8',
			'business'         => $merchant_email, //Email address or account ID of the payment recipient (i.e., the merchant).
			'receiver_email'   => $merchant_email, //Primary email address of the payment recipient (i.e., the merchant
			'order_number'     => $order->order_number,
			"invoice"          => $order->order_number,
			'custom'           => $return_context,
			'item_name'        => $itemname,
			"amount"           => $amount,
			"currency_code"    => $payment_currency,
			/*
					 * 1 – L'adresse spécifiée dans les variables pré-remplies remplace l'adresse de livraison enregistrée auprès de PayPal.
					 * Le payeur voit l'adresse qui est transmise mais ne peut pas la modifier.
					 * Aucune adresse n'est affichée si l'adresse n'est pas valable
					 * (par exemple si des champs requis, tel que le pays, sont manquants) ou pas incluse.
					 * Valeurs autorisées : 0, 1. Valeur par défaut : 0
					 */
			"first_name"       => $order->names,
			"email"            => $order->email,
			"return"           => JROUTE::_ (JURI::root () . 'index.php?option=com_taxibooking&controller=pluginresponse&task=pluginresponsereceived&on=' . $order->order_number . '&pm=' . $order->payment . '&Itemid=' . JRequest::getInt ('Itemid')),
			// Keep this line, needed when testing
			//"return" => JROUTE::_(JURI::root() . 'index.php?option=com_taxibooking&view=pluginresponse&task=pluginnotification&tmpl=component'),
			"notify_url"       => JROUTE::_ (JURI::root () . 'index.php?option=com_taxibooking&controller=pluginresponse&task=pluginnotification&tmpl=component'),
			"cancel_return"    => JROUTE::_ (JURI::root () . 'index.php?option=com_taxibooking&controller=pluginresponse&task=pluginuserpaymentcancel&on=' . $order->order_number . '&pm=' . $order->payment . '&Itemid=' . JRequest::getInt ('Itemid')),
			"rm"               => '2', // the buyer’s browser is redirected to the return URL by using the POST method, and all payment variables are included
			//"pal" => "NRUBJXESJTY24",
			"no_shipping"      => 1,
			"no_note"          => "1");
		
		$url = $this->_getPaypalUrlHttps ($method);

		// add spin image
		$html = '<html><head><meta http-equiv="content-type" content="text/html; charset=utf-8" /><title>Redirection</title></head><body><div style="margin: auto; text-align: center;">';
		
		$tmpl = JRequest::getVar('tmpl', '');
		if($tmpl=='component'){ // if booking form is in iframe, paypal form will be another window
			$html .= '<form action="' . "https://" . $url . '" method="post" name="paypal_form" target="_blank">';
		}
		else {
			$html .= '<form action="' . "https://" . $url . '" method="post" name="paypal_form">';
		}
		
		$html .= '<input type="submit" id="submit_btn" name="submit" value="' . JText::_ ('TBPAYMENT_PAYPAL_REDIRECT_MESSAGE') . '" />';
		foreach ($post_variables as $name => $value) {
			$html .= '<input type="hidden" name="' . $name . '" value="' . htmlspecialchars ($value) . '" />';
		}
		$html .= '</form></div>';
		$html .= ' <script type="text/javascript">';
		$html .= ' document.getElementById("submit_btn").click();';
		$html .= ' </script></body></html>';
		echo $html;
		exit();
	}
	
	/**
	 * This event triggers on Pay Now page, this page is mainly used to make user Pay later for his order
	 */
	public function plgTbOrderPayNow($order){
		
		if (!($method = $this->getPluginMethod ($order->payment))) {
			return NULL; // Another method was selected, do nothing
		}
		if (!$this->selectedThisElement ($method->payment_element)) {
			return FALSE;
		}
		
		$app = JFactory::getApplication();
		$session = JFactory::getSession ();
		$return_context = $session->getId ();
		// Get the page/component configuration
		$elsettings = booking_helper::config();
		
		$merchant_email = $this->_getMerchantEmail ($method);
		if (empty($merchant_email)) {
			$app->enqueueMessage (JText::_ ('TBPAYMENT_PAYPAL_MERCHANT_EMAIL_NOT_SET'));
			return FALSE;
		}
		
		$begin = $order->begin;
		$end = $order->end;
		$order_date = booking_helper::get_order_pickup_date($order, $elsettings);
		
		if($order->booking_type=='hourly'){
			$itemname = JText::sprintf('TBPAYMENT_PAYPAL_HOURLY_ITEM_NAME', $order_date);
		}
		else {
			$itemname = JText::sprintf('TBPAYMENT_PAYPAL_ADDRESS_ITEM_NAME', $begin, $end, $order_date);
		}
		
		if ($order->returntrip==1)
		{
			$order_return_date = booking_helper::get_order_return_date($order, $elsettings);
			$itemname .= JText::sprintf('TBPAYMENT_PAYPAL_ITEM_NAME_RETURN', $order_return_date);
		}
		
		$prepay = $this->_getPrePaymentPercent( $method );
		$amount = $order->cprice * $prepay / 100;
		
		$payment_currency = $elsettings->currency;
		// convert currency
		if((int)$method->convert_currency==1 && $method->convert_to!=""){
			$amount = $amount * (float)$method->conversion_index;
			$payment_currency = $method->convert_to;
			
			// Hungarian Forint = HUF, Japanese Yen = JPY, Taiwan New Dolla = TWD 
			// Note Decimal amounts are not supported for these currency. Passing a decimal amount will throw an error.
			$special = array('HUF', 'JPY', 'TWD');
			if(in_array($method->convert_to, $special)){
				$amount = round($amount);
			}
			else {
				$amount = number_format($amount, 2, '.', '');
			}
		}
		else {
			$amount = number_format($amount, 2, '.', '');
		}

		$post_variables = Array(
			'cmd'              => '_ext-enter',
			'redirect_cmd'     => '_xclick',
			'upload'           => '1', //Indicates the use of third-party shopping cart
			'charset'	   => 'utf-8',
			'business'         => $merchant_email, //Email address or account ID of the payment recipient (i.e., the merchant).
			'receiver_email'   => $merchant_email, //Primary email address of the payment recipient (i.e., the merchant
			'order_number'     => $order->order_number,
			"invoice"          => $order->order_number,
			'custom'           => $return_context,
			'item_name'        => $itemname,
			"amount"           => $amount,
			"currency_code"    => $payment_currency,
			/*
					 * 1 – L'adresse spécifiée dans les variables pré-remplies remplace l'adresse de livraison enregistrée auprès de PayPal.
					 * Le payeur voit l'adresse qui est transmise mais ne peut pas la modifier.
					 * Aucune adresse n'est affichée si l'adresse n'est pas valable
					 * (par exemple si des champs requis, tel que le pays, sont manquants) ou pas incluse.
					 * Valeurs autorisées : 0, 1. Valeur par défaut : 0
					 */
			"first_name"       => $order->names,
			"email"            => $order->email,
			"return"           => JROUTE::_(JURI::root () . 'index.php?option=com_taxibooking&controller=pluginresponse&task=pluginresponsereceived&on=' . $order->order_number . '&pm=' . $order->payment . '&Itemid=' . JRequest::getInt ('Itemid')),
			// Keep this line, needed when testing
			//"return" => JROUTE::_(JURI::root() . 'index.php?option=com_taxibooking&view=pluginresponse&task=pluginnotification&tmpl=component'),
			"notify_url"       => JROUTE::_(JURI::root () . 'index.php?option=com_taxibooking&controller=pluginresponse&task=pluginnotification&tmpl=component'),
			"cancel_return"    => JROUTE::_(JURI::root () . 'index.php?option=com_taxibooking&controller=pluginresponse&task=pluginuserpaymentcancel&on=' . $order->order_number . '&pm=' . $order->payment . '&Itemid=' . JRequest::getInt ('Itemid')),
			"rm"               => '2', // the buyer’s browser is redirected to the return URL by using the POST method, and all payment variables are included
			//"pal" => "NRUBJXESJTY24",
			"no_shipping"      => 1,
			"no_note"          => "1");
		
		$url = $this->_getPaypalUrlHttps ($method);

		// add spin image
		$html = '<html><head><meta http-equiv="content-type" content="text/html; charset=utf-8" /><title>Redirection</title></head><body><div style="margin: auto; text-align: center;">';
		
		$tmpl = JRequest::getVar('tmpl', '');
		if($tmpl=='component'){ // if booking form is in iframe, paypal form will be another window
			$html .= '<form action="' . "https://" . $url . '" method="post" name="paypal_form" target="_blank">';
		}
		else {
			$html .= '<form action="' . "https://" . $url . '" method="post" name="paypal_form">';
		}
		
		$html .= '<input type="submit" id="submit_btn" name="submit" value="' . JText::_ ('TBPAYMENT_PAYPAL_REDIRECT_MESSAGE') . '" />';
		foreach ($post_variables as $name => $value) {
			$html .= '<input type="hidden" name="' . $name . '" value="' . htmlspecialchars ($value) . '" />';
		}
		$html .= '</form></div>';
		$html .= ' <script type="text/javascript">';
		$html .= ' document.getElementById("submit_btn").click();';
		$html .= ' </script></body></html>';
		echo $html;
		exit();
	}
	
	/**
	 * @param $method
	 * @return mixed
	 */
	private function _getMerchantEmail ($method) {

		return $method->sandbox ? $method->sandbox_merchant_email : $method->paypal_merchant_email;
	}
	
	/**
	 * @param $method
	 * @return mixed
	 */
	private function _getPrePaymentPercent ($method) {

		return ( '' == $method->prepayment_percent ) ? 100 : $method->prepayment_percent;
	}

	/**
	 * @param $method
	 * @return string
	 */
	private function _getPaypalUrl ($method) {

		$url = $method->sandbox ? 'www.sandbox.paypal.com' : 'www.paypal.com';

		return $url;
	}

	/**
	 * @param $method
	 * @return string
	 */
	private function _getPaypalUrlHttps ($method) {

		$url = $this->_getPaypalUrl ($method);
		$url = $url . '/cgi-bin/webscr';

		return $url;
	}

	/*
		 * CheckPaypalIPs
		 * Cannot be checked with Sandbox
		 * From VM1.1
		 */

	/**
	 * @param $test_ipn
	 * @return mixed
	 */
	private function checkPaypalIps ($test_ipn,$invoice) {

		//return;
		// Get the list of IP addresses for www.paypal.com and notify.paypal.com
		$paypal_iplist = gethostbynamel ('www.paypal.com');
		$paypal_iplist2 = gethostbynamel ('notify.paypal.com');
		$paypal_iplist3 = array('216.113.188.202', '216.113.188.203', '216.113.188.204', '66.211.170.66');
		$paypal_iplist = array_merge ($paypal_iplist, $paypal_iplist2, $paypal_iplist3);

		$paypal_sandbox_hostname = 'ipn.sandbox.paypal.com';
		$remote_hostname = gethostbyaddr ($_SERVER['REMOTE_ADDR']);

		$valid_ip = FALSE;

		if ($paypal_sandbox_hostname == $remote_hostname) {
			$valid_ip = TRUE;
			$hostname = 'www.sandbox.paypal.com';
		} else {
			$ips = "";
			// Loop through all allowed IPs and test if the remote IP connected here
			// is a valid IP address
			if (in_array ($_SERVER['REMOTE_ADDR'], $paypal_iplist)) {
				$valid_ip = TRUE;
			}
			$hostname = 'www.paypal.com';
		}

		if (!$valid_ip) {
			
			$mainframe = JFactory::getApplication();
			$SiteName = $mainframe->getCfg('sitename');
			$MailFrom = $mainframe->getCfg('mailfrom');
			$FromName = $mainframe->getCfg('fromname');

			$mailsubject = "PayPal IPN Transaction on your site: Possible fraud";
			$mailbody = "Error code 506. Possible fraud. Error with REMOTE IP ADDRESS = " . $_SERVER['REMOTE_ADDR'] . ".
                        The remote address of the script posting to this notify script does not match a valid PayPal ip address\n
			The Order ID received was: $invoice";
	    
			// send emails to all admin
			$db = JFactory::getDBO();
			$query = 'SELECT u.id, u.email, u.name' .
				' FROM #__users AS u' .
				' LEFT JOIN #__user_usergroup_map AS gu ON gu.user_id = u.id' .
				' WHERE (gu.group_id = 8 OR gu.group_id = 7) ' .
				' AND u.sendEmail = 1 AND u.block = 0';
			$db->setQuery($query);
			if (!$db->query()) {
			    JError::raiseError(500, $db->stderr(true));
			    return;
			}
			$adminRows = $db->loadObjectList();
			
			foreach ($adminRows as $adminRow) {
		
			    $mail =  JFactory::getMailer();
			    $mail->setSender(array($MailFrom, $FromName));
			    $mail->setSubject($mailsubject);
			    $mail->setBody($mailbody);
			    $mail->IsHTML(true);
			    $mail->addRecipient($adminRow->email);
			    $sent = $mail->Send();
			}
			print $mailbody;
			exit();
		}

		/*if (!($hostname == "www.sandbox.paypal.com" && $test_ipn == 1)) {
			$res = "FAILED";
			$mailsubject = "PayPal Sandbox Transaction";
			$mailbody = "Hello,
		A fatal error occurred while processing a paypal transaction.
		----------------------------------
		Hostname: $hostname
		URI: $uri
		A Paypal transaction was made using the sandbox without your site in Paypal-Debug-Mode";
			//vmMail($mosConfig_mailfrom, $mosConfig_fromname, $debug_email_address, $mailsubject, $mailbody );
			exit();
		}*/
	}
	
	/**
	 * @param $html
	 * @return bool|null|string
	 */
	function plgTbOnPaymentResponseReceived (&$html) {

		$paymentmethod_id = JRequest::getInt ('pm', 0);
		$order_number = JRequest::getString ('on', 0);

		if (!($method = $this->getPluginMethod ($paymentmethod_id))) {
			return NULL; // Another method was selected, do nothing
		}
		if (!$this->selectedThisElement ($method->payment_element)) {
			return NULL;
		}
		if (!($order = booking_helper::get_order_by_order_number($order_number))) {
			return NULL;
		}
		if (!($paymentTable = $this->getDataByOrderId ($order->id))) {
			// JError::raiseWarning(500, $db->getErrorMsg());
			return '';
		}
		$payment_name = $this->renderPluginName ($method);
		
		return TRUE;
	}

	/**
	 * @return bool|null
	 */
	function plgTbOnUserPaymentCancel () {
		
		$order_number = JRequest::getString ('on', '');
		$paymentmethod_id = JRequest::getInt ('pm', '');
		if (empty($order_number) or empty($paymentmethod_id) or !$this->selectedThisByMethodId ($paymentmethod_id)) {
			return NULL;
		}
		if (!($order = booking_helper::get_order_by_order_number($order_number))) {
			return NULL;
		}
		if (!($paymentTable = $this->getDataByOrderId($order->id))) {
			return NULL;
		}

		$session = JFactory::getSession ();
		$return_context = $session->getId ();
		if (strcmp ($paymentTable[0]->paypal_custom, $return_context) === 0) {
			$this->handlePaymentUserCancel ($order->id);
		}
		return TRUE;
	}

	/*
	*   This event is fired by Offline Payment. It can be used to validate the payment data as entered by the user.
	*/
	function plgTbOnPaymentNotification () {

		$paypal_data = JRequest::get ('post');
		
		/*$fp = fopen(JPATH_ROOT.DS.'plugins'.DS.'tbpayment'.DS.'paypal'.DS.'paypal_response.txt', 'a');
		fwrite($fp, 'PayPal Response: Date - '.date('Y-m-d H:i:s')."\n");
		fwrite($fp, print_r($paypal_data, true));
		fclose($fp);*/

		if (!isset($paypal_data['invoice'])) {
			return NULL;
		}
		$order_number = $paypal_data['invoice'];
		if (!($order = booking_helper::get_order_by_order_number($order_number))) {
			return NULL;
		}
		if (!($payments = $this->getDataByOrderId($order->id))) {
			return NULL;
		}

		$method = $this->getPluginMethod ($payments[0]->paymentmethod_id);
		if (!$this->selectedThisElement ($method->payment_element)) {
			return FALSE;
		}
	
		$this->checkPaypalIps($paypal_data['test_ipn'], $order_number);
		
		$order_data = array();
		
		// _processIPN checks that  $res== "VERIFIED"
		/*$error_msg = $this->_processIPN ($paypal_data, $method);

		if (!(empty($error_msg))) {
			return NULL;
		}*/
		/*
		 * https://cms.paypal.com/us/cgi-bin/?cmd=_render-content&content_ID=developer/e_howto_html_IPNandPDTVariables
		 * The status of the payment:
		 * Canceled_Reversal: A reversal has been canceled. For example, you won a dispute with the customer, and the funds for the transaction that was reversed have been returned to you.
		 * Completed: The payment has been completed, and the funds have been added successfully to your account balance.
		 * Created: A German ELV payment is made using Express Checkout.
		 * Denied: You denied the payment. This happens only if the payment was previously pending because of possible reasons described for the pending_reason variable or the Fraud_Management_Filters_x variable.
		 * Expired: This authorization has expired and cannot be captured.
		 * Failed: The payment has failed. This happens only if the payment was made from your customer’s bank account.
		 * Pending: The payment is pending. See pending_reason for more information.
		 * Refunded: You refunded the payment.
		 * Reversed: A payment was reversed due to a chargeback or other type of reversal. The funds have been removed from your account balance and returned to the buyer. The reason for the reversal is specified in the ReasonCode element.
		 * Processed: A payment has been accepted.
		 * Voided: This authorization has been voided.
		 *
		 */
		if (empty($paypal_data['payment_status']) || ($paypal_data['payment_status'] != 'Completed' && $paypal_data['payment_status'] != 'Pending')) {
			//return false;
		}
		$lang = JFactory::getLanguage();
		$lang->load('plg_tbpayment_' . $this->_name,JPATH_ADMINISTRATOR);

		// 1. check the payment_status is Completed
		if (strcmp ($paypal_data['payment_status'], 'Completed') == 0) {
			
			// 2. check that txn_id has not been previously processed
			if ($this->_check_txn_id_already_processed ($payments, $paypal_data['txn_id'])) {
				return;
			}
			// 3. check email and amount currency is correct
			if (!$this->_check_email_amount_currency ($payments, $this->_getMerchantEmail ($method), $paypal_data)) {
				return;
			}
			
			// now we can process the payment
			$order_data['order_status'] = $method->status_success;
		}
		elseif (strcmp ($paypal_data['payment_status'], 'Pending') == 0) {
			$key = 'TBPAYMENT_PAYPAL_PENDING_REASON_FE_' . strtoupper ($paypal_data['pending_reason']);
			if (!$lang->hasKey ($key)) {
				$key = 'TBPAYMENT_PAYPAL_PENDING_REASON_FE_DEFAULT';
			}
			$order_data['order_status'] = $method->status_pending;
		}
		elseif (isset ($paypal_data['payment_status'])) {
			$order_data['order_status'] = $method->status_canceled;
		}
		else {
			
		}
		
		$this->_storePaypalInternalData ($method, $paypal_data, $order->id, $payments[0]->paymentmethod_id);
		booking_helper::update_order_status($order->id, $order_data, true); // true = notify admins
	}
	
	/**
	 * Get ipn data, send verification to PayPal, run corresponding handler
	 *
	 * @param array $data
	 * @return string Empty string if data is valid and an error message otherwise
	 * @access protected
	 */
	function _processIPN ($paypal_data, $method) {

		$secure_post = false;
		$paypal_url = $this->_getPaypalURL ($method);
		// read the post from PayPal system and add 'cmd'
		$post_msg = 'cmd=_notify-validate';
		foreach ($paypal_data as $key => $value) {
			if ($key != 'view' && $key != 'layout') {
				$value = urlencode ($value);
				$post_msg .= "&$key=$value";
			}
		}

		$this->checkPaypalIps ($paypal_data['test_ipn']);

		// post back to PayPal system to validate
		$header = "POST /cgi-bin/webscr HTTP/1.0\r\n";
		$header .= "Content-Type: application/x-www-form-urlencoded\r\n";
		$header .= "Content-Length: " . strlen ($post_msg) . "\r\n\r\n";

		if ($secure_post) {
			// If possible, securely post back to paypal using HTTPS
			// Your PHP server will need to be SSL enabled
			$fps = fsockopen ('ssl://' . $paypal_url, 443, $errno, $errstr, 30);
		} else {
			$fps = fsockopen ($paypal_url, 80, $errno, $errstr, 30);
		}

		if (!$fps) {
			return JText::sprintf ('TBPAYMENT_PAYPAL_ERROR_POSTING_IPN', $errstr, $errno); // send email
		} else {
			fputs ($fps, $header . $post_msg);
			while (!feof ($fps)) {
				$res = fgets ($fps, 1024);

				if (strcmp ($res, 'VERIFIED') == 0) {
					return '';
				} elseif (strcmp ($res, 'INVALID') == 0) {
					$emailBody = "Hello,\n\nerror with paypal IPN NOTIFICATION" . " " . $res . "\n";
					// If 'INVALID', send an email. TODO: Log for manual investigation.
					foreach ($paypal_data as $key => $value) {
						$emailBody .= $key . " = " . $value . "\n";
					}
					return JText::_ ('TBPAYMENT_PAYPAL_ERROR_IPN_VALIDATION') . $res;
				}
			}
		}

		fclose ($fps);
		return '';
	}
	function _check_txn_id_already_processed ($payments, $txn_id) {

		foreach ($payments as $payment) {
			if ($payment->paypal_response_txn_id == $txn_id) {
				return TRUE;
			}
		}

		return FALSE;
	}
	function _check_email_amount_currency ($payments, $email, $paypal_data) {

		/*
		 * TODO Not checking yet because config do not have primary email address
		* Primary email address of the payment recipient (that is, the merchant).
		* If the payment is sent to a non-primary email address on your PayPal account,
		* the receiver_email is still your primary email.
		*/
		/*
		if ($payments[0]->payment_order_total==$email) {
			return true;
		}
		*/
		$prepay = $payments[0]->prepayment_percent;
		if($prepay<100){
			$payments[0]->payment_order_total = $payments[0]->payment_order_total*$prepay/100;
		}
		
		if ($payments[0]->payment_order_total == $paypal_data['mc_gross']) {
			return TRUE;
		}
		if ($payments[0]->payment_currency == $paypal_data['mc_currency']) {
			return TRUE;
		}
		return FALSE;
	}
	/**
	 * @param $method
	 * @param $paypal_data
	 */
	function _storePaypalInternalData ($method, $paypal_data, $order_id, $paymentmethod_id) {

		// get all know columns of the table
		$columns = array();
		$db = JFactory::getDBO ();
		$query = 'SHOW COLUMNS FROM `' . $this->_tablename . '` ';
		$db->setQuery ($query);
		$rows = $db->loadObjectList();
		if(!empty($rows)){
			foreach($rows as $row){
				$columns[] = $row->Field;
			}
		}
		$post_msg = '';
		foreach ($paypal_data as $key => $value) {
			$post_msg .= $key . "=" . $value . "<br />";
			$table_key = 'paypal_response_' . $key;
			if (in_array ($table_key, $columns)) {
				$response_fields[$table_key] = $value;
			}
		}

		//$response_fields[$this->_tablepkey] = $this->_getTablepkeyValue($order_id);
		$response_fields['payment_name'] = $this->renderPluginName ($method);
		$response_fields['paypalresponse_raw'] = $post_msg;
		$response_fields['order_number'] = $paypal_data['invoice'];
		$response_fields['order_id'] = $order_id;
		$response_fields['paymentmethod_id'] = $paymentmethod_id;
		$response_fields['paypal_custom'] = $paypal_data['custom'];

		//$preload=true   preload the data here too preserve not updated data
		$this->storePluginInternalData ($response_fields);
	}
	
	/**
	 * Display stored payment data for an order
	 */
	function plgTbOnShowOrderBEPayment ($order_id, $payment_method_id) {

		if (!$this->selectedThisByMethodId ($payment_method_id)) {
			return NULL; // Another method was selected, do nothing
		}

		if (!($payments = $this->_getPaypalInternalData($order_id))) {
			// JError::raiseWarning(500, $db->getErrorMsg());
			return '';
		}

		$html = '<table class="adminlist" width="50%">' . "\n";
		$html .= '<thead>
<tr>
<th class="key" colspan="2" style="text-align: center;">Payment Method</th>
</tr>
</thead>';
		$code = "paypal_response_";
		$first = TRUE;
		foreach ($payments as $payment) {
			$html .= '<tr class="row1"><td>' . JText::_ ('TBPAYMENT_PAYPAL_DATE') . '</td><td align="left">' . $payment->created_on . '</td></tr>';
			// Now only the first entry has this data when creating the order
			if ($first) {
				$html .= '<tr>
<td class="key">' . JText::_ ('TBPAYMENT_PAYPAL_NAME') . '</td>
<td align="left">
<span class="payment_name">'.$payment->payment_name.'</span>
<br>
</td>
</tr>';
				// keep that test to have it backwards compatible. Old version was deleting that column  when receiving an IPN notification
				if ($payment->payment_order_total and  $payment->payment_order_total !=  0.00) {
					$html .= '<tr>
<td class="key">' . JText::_ ('TBPAYMENT_PAYPAL_PAYMENT_TOTAL_CURRENCY') . '</td>
<td align="left">
<span class="payment_name">'.$payment->payment_order_total . ' ' . $payment->payment_currency.'</span>
<br>
</td>
</tr>';
					$prepay = $payment->prepayment_percent;
					if($prepay<100){
						$html .= '<tr>
						<td class="key">' . JText::_ ('TBPAYMENT_PAYPAL_PRE_PAYMENT_TOTAL') . '</td>
						<td align="left">
						<span class="payment_name">'.number_format($payment->payment_order_total*$prepay/100, 2, '.', ',') . ' ' . $payment->payment_currency.'</span>
						<br>
						</td>
						</tr>';
					}
				}
				$first = FALSE;
			}
			foreach ($payment as $key => $value) {
				// only displays if there is a value or the value is different from 0.00 and the value
				if ($value) {
					if (substr ($key, 0, strlen ($code)) == $code) {
						$html .= '<tr>
								<td class="key">'.$key.'</td>
								<td align="left">'.$value.'</td>
							</tr>';
					}
				}
			}
		}
		$html .= '</table>' . "\n";
		return $html;
	}
	
	/**
	 * Display stored payment data in confirmation emails and invoice
	 */
	function plgTbOnShowOrderEmailsInvoice ($order_id, $payment_method_id) {

		if (!($method = $this->selectedThisByMethodId ($payment_method_id))) {
			return NULL; // Another method was selected, do nothing
		}

		if (!($payments = $this->_getPaypalInternalData($order_id))) {
			// JError::raiseWarning(500, $db->getErrorMsg());
			return '';
		}

		$html = '';
		
		$first = TRUE;
		foreach ($payments as $payment) {
			
			// Now only the first entry has this data when creating the order
			if ($first) {
				$html .= '<tr>
<td width="35%">' . JText::_ ('TBPAYMENT_PAYPAL_PAYMENT_NAME') . ':</td>
<td width="35%">'.$payment->payment_name.'</td>
</tr>';
				// keep that test to have it backwards compatible. Old version was deleting that column  when receiving an IPN notification
				if ($payment->payment_order_total and  $payment->payment_order_total !=  0.00)
				{
					$prepay = $payment->prepayment_percent;
					$payment_order_total = $payment->payment_order_total;
					
					if($prepay<100){
						
						$converted_currency_lbl = '';
						if((int)$method->convert_currency==1 && $method->convert_to!=""){
							$payment_order_total = $payment_order_total/(float)$method->conversion_index;
							$due_now = $payment_order_total * $prepay / 100;
							$due_later = $payment_order_total - $due_now;
							
							$converted = $due_now *(float)$method->conversion_index;
							$payment_currency = $method->convert_to;
					
							// Hungarian Forint = HUF, Japanese Yen = JPY, Taiwan New Dolla = TWD 
							// Note Decimal amounts are not supported for these currency. Passing a decimal amount will throw an error.
							$special = array('HUF', 'JPY', 'TWD');
							if(in_array($method->convert_to, $special)){
								$converted = round($converted);
							}
							else {
								$converted = number_format($converted, 2);
							}
							$converted_currency_lbl .= JText::sprintf($this->currencies[$payment_currency]['format'], $converted);
						}
						else {
							$due_now = $payment_order_total * $prepay / 100;
							$due_later = $payment_order_total - $due_now;
						}
						
					$html .= '<tr>
							<td width="35%">'.JText::_('TBPAYMENT_PAYPAL_PREPAYMENT_DUE_NOW').':</td>
							<td width="35%"><strong>'.booking_helper::price_display($due_now).$converted_currency_lbl.'</strong></td>
						</tr>';
					$html .= '<tr>
							<td width="35%">'.JText::_('TBPAYMENT_PAYPAL_PREPAYMENT_DUE_LATER').':</td>
							<td width="35%"><strong>'.booking_helper::price_display($due_later).'</strong></td>
						</tr>';
					}
				}
				$first = FALSE;
			}
		}
		return $html;
	}

	/**
	 * @param        $order_id
	 * @param string $order_number
	 * @return mixed|string
	 */
	function _getPaypalInternalData ($order_id, $order_number = '') {

		$db = JFactory::getDBO ();
		$q = 'SELECT * FROM `' . $this->_tablename . '` WHERE ';
		if ($order_number) {
			$q .= " `order_number` = '" . $order_number . "'";
		} else {
			$q .= ' `order_id` = ' . $order_id;
		}

		$db->setQuery ($q);
		if (!($payments = $db->loadObjectList ())) {
			// JError::raiseWarning(500, $db->getErrorMsg());
			return '';
		}
		return $payments;
	}
}